(*
 * Copyright (c) 2016 Vittorio Giovara <vittorio.giovara@gmail.com>
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *)

(**
 * @file
 * @ingroup lavu_video_spherical
 * Spherical video
 *)

(*
 * FFVCL - Delphi FFmpeg VCL Components
 * http://www.DelphiFFmpeg.com
 *
 * Original file: libavutil/spherical.h
 * Ported by CodeCoolie@CNSW 2023/11/28 -> $Date:: 2025-01-17 #$
 *)

(*
FFmpeg Delphi/Pascal Headers and Examples License Agreement

A modified part of FFVCL - Delphi FFmpeg VCL Components.
Copyright (c) 2008-2025 DelphiFFmpeg.com
All rights reserved.
http://www.DelphiFFmpeg.com

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

This source code is provided "as is" by DelphiFFmpeg.com without
warranty of any kind, either expressed or implied, including but not
limited to the implied warranties of merchantability and/or fitness
for a particular purpose.

Please also notice the License agreement of FFmpeg libraries.
*)

unit libavutil_spherical;

interface

{$I CompilerDefines.inc}

uses
  FFTypes;

{$I libversion.inc}

(**
 * @defgroup lavu_video_spherical Spherical video mapping
 * @ingroup lavu_video
 *
 * A spherical video file contains surfaces that need to be mapped onto a
 * sphere. Depending on how the frame was converted, a different distortion
 * transformation or surface recomposition function needs to be applied before
 * the video should be mapped and displayed.
 * @{
 *)

(**
 * Projection of the video surface(s) on a sphere.
 *)
type
  TAVSphericalProjection = (
    (**
     * Video represents a sphere mapped on a flat surface using
     * equirectangular projection.
     *)
    AV_SPHERICAL_EQUIRECTANGULAR,

    (**
     * Video frame is split into 6 faces of a cube, and arranged on a
     * 3x2 layout. Faces are oriented upwards for the front, left, right,
     * and back faces. The up face is oriented so the top of the face is
     * forwards and the down face is oriented so the top of the face is
     * to the back.
     *)
    AV_SPHERICAL_CUBEMAP,

    (**
     * Video represents a portion of a sphere mapped on a flat surface
     * using equirectangular projection. The @ref bounding fields indicate
     * the position of the current video in a larger surface.
     *)
    AV_SPHERICAL_EQUIRECTANGULAR_TILE,

    (**
     * Video frame displays as a 180 degree equirectangular projection.
     *)
    AV_SPHERICAL_HALF_EQUIRECTANGULAR,

    (**
     * Video frame displays on a flat, rectangular 2D surface.
     *)
    AV_SPHERICAL_RECTILINEAR,

    (**
     * Fisheye projection (Apple).
     * See: https://developer.apple.com/documentation/coremedia/cmprojectiontype/fisheye
     *)
    AV_SPHERICAL_FISHEYE
  );

(**
 * This structure describes how to handle spherical videos, outlining
 * information about projection, initial layout, and any other view modifier.
 *
 * @note The struct must be allocated with av_spherical_alloc() and
 *       its size is not a part of the public ABI.
 *)
  PAVSphericalMapping = ^TAVSphericalMapping;
  TAVSphericalMapping = record
    (**
     * Projection type.
     *)
    projection: TAVSphericalProjection;

    (**
     * @name Initial orientation
     * @{
     * There fields describe additional rotations applied to the sphere after
     * the video frame is mapped onto it. The sphere is rotated around the
     * viewer, who remains stationary. The order of transformation is always
     * yaw, followed by pitch, and finally by roll.
     *
     * The coordinate system matches the one defined in OpenGL, where the
     * forward vector (z) is coming out of screen, and it is equivalent to
     * a rotation matrix of R = r_y(yaw) * r_x(pitch) * r_z(roll).
     *
     * A positive yaw rotates the portion of the sphere in front of the viewer
     * toward their right. A positive pitch rotates the portion of the sphere
     * in front of the viewer upwards. A positive roll tilts the portion of
     * the sphere in front of the viewer to the viewer's right.
     *
     * These values are exported as 16.16 fixed point.
     *
     * See this equirectangular projection as example:
     *
     * @code{.unparsed}
     *                   Yaw
     *     -180           0           180
     *   90 +-------------+-------------+  180
     *      |             |             |                  up
     * P    |             |             |                 y|    forward
     * i    |             ^             |                  |   /z
     * t  0 +-------------X-------------+    0 Roll        |  /
     * c    |             |             |                  | /
     * h    |             |             |                 0|/_____right
     *      |             |             |                        x
     *  -90 +-------------+-------------+ -180
     *
     * X - the default camera center
     * ^ - the default up vector
     * @endcode
     *)
    yaw: Cardinal;   ///< Rotation around the up vector [-180, 180].
    pitch: Cardinal; ///< Rotation around the right vector [-90, 90].
    roll: Cardinal;  ///< Rotation around the forward vector [-180, 180].
    (**
     * @}
     *)

    (**
     * @name Bounding rectangle
     * @anchor bounding
     * @{
     * These fields indicate the location of the current tile, and where
     * it should be mapped relative to the original surface. They are
     * exported as 0.32 fixed point, and can be converted to classic
     * pixel values with av_spherical_bounds().
     *
     * @code{.unparsed}
     *      +----------------+----------+
     *      |                |bound_top |
     *      |            +--------+     |
     *      | bound_left |tile    |     |
     *      +<---------->|        |<--->+bound_right
     *      |            +--------+     |
     *      |                |          |
     *      |    bound_bottom|          |
     *      +----------------+----------+
     * @endcode
     *
     * If needed, the original video surface dimensions can be derived
     * by adding the current stream or frame size to the related bounds,
     * like in the following example:
     *
     * @code{c}
     *     original_width  = tile->width  + bound_left + bound_right;
     *     original_height = tile->height + bound_top  + bound_bottom;
     * @endcode
     *
     * @note These values are valid only for the tiled equirectangular
     *       projection type (@ref AV_SPHERICAL_EQUIRECTANGULAR_TILE),
     *       and should be ignored in all other cases.
     *)
    bound_left: Cardinal;   ///< Distance from the left edge
    bound_top: Cardinal;    ///< Distance from the top edge
    bound_right: Cardinal;  ///< Distance from the right edge
    bound_bottom: Cardinal; ///< Distance from the bottom edge
    (**
     * @}
     *)

    (**
     * Number of pixels to pad from the edge of each cube face.
     *
     * @note This value is valid for only for the cubemap projection type
     *       (@ref AV_SPHERICAL_CUBEMAP), and should be ignored in all other
     *       cases.
     *)
    padding: Cardinal;
  end;

(**
 * Allocate a AVSphericalVideo structure and initialize its fields to default
 * values.
 *
 * @return the newly allocated struct or NULL on failure
 *)
function av_spherical_alloc(size: PSize_t): PAVSphericalMapping; cdecl; external AVUTIL_LIBNAME name _PU + 'av_spherical_alloc';

(**
 * Convert the @ref bounding fields from an AVSphericalVideo
 * from 0.32 fixed point to pixels.
 *
 * @param map    The AVSphericalVideo map to read bound values from.
 * @param width  Width of the current frame or stream.
 * @param height Height of the current frame or stream.
 * @param left   Pixels from the left edge.
 * @param top    Pixels from the top edge.
 * @param right  Pixels from the right edge.
 * @param bottom Pixels from the bottom edge.
 *)
procedure av_spherical_tile_bounds(const map: PAVSphericalMapping;
                              width, height: Size_t;
                              left, top, right, bottom: PSize_t); cdecl; external AVUTIL_LIBNAME name _PU + 'av_spherical_tile_bounds';

(**
 * Provide a human-readable name of a given AVSphericalProjection.
 *
 * @param projection The input AVSphericalProjection.
 *
 * @return The name of the AVSphericalProjection, or "unknown".
 *)
function av_spherical_projection_name(projection: TAVSphericalProjection): PAnsiChar; cdecl; external AVUTIL_LIBNAME name _PU + 'av_spherical_projection_name';

(**
 * Get the AVSphericalProjection form a human-readable name.
 *
 * @param name The input string.
 *
 * @return The AVSphericalProjection value, or -1 if not found.
 *)
function av_spherical_from_name(const name: PAnsiChar): Integer; cdecl; external AVUTIL_LIBNAME name _PU + 'av_spherical_from_name';

(**
 * @}
 *)

implementation

end.
