unit SimpleConverterFrm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ComCtrls, FFBaseComponent, FFEncode, FFLoad, FFLog;

type
  TfrmSimpleConverter = class(TForm)
    lblStatus: TLabel;
    btnOpen: TButton;
    btnStart: TButton;
    btnStop: TButton;
    btnPause: TButton;
    btnResume: TButton;
    btnWebSite: TButton;
    cboLogLevel: TComboBox;
    chkPreview: TCheckBox;
    mmoLog: TMemo;
    ProgressBar1: TProgressBar;
    FFEncoder: TFFEncoder;
    FFLogger: TFFLogger;
    OpenDialog1: TOpenDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure btnOpenClick(Sender: TObject);
    procedure btnStartClick(Sender: TObject);
    procedure btnStopClick(Sender: TObject);
    procedure btnPauseClick(Sender: TObject);
    procedure btnResumeClick(Sender: TObject);
    procedure btnWebSiteClick(Sender: TObject);
    procedure cboLogLevelChange(Sender: TObject);
    procedure chkPreviewClick(Sender: TObject);
    procedure FFEncoderProgress(Sender: TObject; AProgressInfo: PProgressInfo);
    procedure FFEncoderTerminate(Sender: TObject; const ATerminateInfo: TTerminateInfo);
    procedure FFLoggerLog(Sender: TObject; AThreadID: Integer;
      ALogLevel: TLogLevel; const ALogMsg: string);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  frmSimpleConverter: TfrmSimpleConverter;

implementation

{$R *.dfm}

uses
  ShellAPI,
  MyUtils;

const
  CLibAVPath = 'LibAV';

  SAppTitle = 'Demo of FFEncoder %s';
  SCaption = 'Demo of FFEncoder %s (SimpleConverter) - Delphi FFmpeg VCL Components';
  SWebSiteC = 'http://www.CCAVC.com';
  SWebSiteE = 'http://www.DelphiFFmpeg.com';


  CDialogOptions = [ofHideReadOnly, ofFileMustExist, ofEnableSizing];
  CPictureFiles = '*.BMP;*.GIF;*.JPEG;*.JPG;*.PNG;';
  CAudioFiles = '*.AAC;*.AC3;*.APE;*.DTS;*.FLAC;*.M4A;*.MKA;*.MP2;*.MP3;' +
      '*.MPA;*.MPC;*.OFR;*.OGG;*.RA;*.TTA;*.WAV;*.WMA;';
  CVideoFiles = '*.3GP;*.ASF;*.AVI;*.AVM;*.AVS;*.DAT;*.FLV;*.MKV;*.MOV;' +
      '*.MP4;*.MPEG;*.MPG;*.NSV;*.OGM;*.RM;*.RMVB;*.TP;*.TS;*.VOB;*.WMV;';
  CDialogFilter =
      'Video/Audio/Picture Files|' + CVideoFiles + CAudioFiles + CPictureFiles +
      '|Video Files|' + CVideoFiles +
      '|Audio Files|' + CAudioFiles +
      '|Picture Files|' + CPictureFiles +
      '|All Files|*.*';

var
  SWebSite: string = SWebSiteE;

function GenerateOutputFileName(const AFileName, AFileExt: string): string;
var
  LBaseName: string;
  I: Integer;
begin
  // generate output filename automatically
  LBaseName := ChangeFileExt(AFileName, '');
  Result := LBaseName + '_(new)' + AFileExt;
  if FileExists(Result) then
  begin
    I := 1;
    while FileExists(LBaseName + '_(new_' + IntToStr(I) + ')' + AFileExt) do
      Inc(I);
    Result := LBaseName + '_(new_' + IntToStr(I) + ')' + AFileExt;
  end;
end;

procedure TfrmSimpleConverter.FormCreate(Sender: TObject);
begin
  Application.Title := Format(SAppTitle, [FFEncoder.Version]);
  Self.Caption := Format(SCaption, [FFEncoder.Version]);

  if SysUtils.SysLocale.PriLangID = LANG_CHINESE then
    SWebSite := SWebSiteC
  else
    SWebSite := SWebSiteE;

  mmoLog.Text := SWebSite + #13#10#13#10;
  btnWebsite.Hint := SWebSite;
  btnWebsite.ShowHint := True;

  // open dialog setting
  OpenDialog1.Options := CDialogOptions;
  OpenDialog1.Filter := CDialogFilter;

  // disable all fpu exceptions(floating point exceptions):
  //   invalid operation, denormalized, divide by zero, overflow, underflow, inexact/precision
  FFEncoder.DisableFPUExceptions;
end;

procedure TfrmSimpleConverter.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  with FFEncoder do
  begin
    // Clear the event handlers
    OnProgress := nil;
    OnTerminate := nil;

    // Break converting
    Stop;
  end;
  FFLogger.OnLog := nil;
end;

procedure TfrmSimpleConverter.btnOpenClick(Sender: TObject);
var
  LIndex: Integer;
  LOptions: string;
  LInFileName: string;
  LOutFileName: string;
begin
  // Load dynamic link libraries
  if not FFEncoder.AVLibLoaded then
  begin
    // TPathFileName = type WideString;
    // FFEncoder.LoadAVLib(const APath: TPathFileName): Boolean;
    // APath: Full path indicates location of FFmpeg DLLs.
    //        It can be empty, let Windows search DLLs in current dir or environment <PATH>
    //if not FFEncoder.LoadAVLib(ExtractFilePath(Application.ExeName) + CLibAVPath) then
    // the routine ExePath() is implemented in unit MyUtils which returns WideString type
    // of ExtractFilePath(Application.ExeName)
    if not FFEncoder.LoadAVLib(ExePath + CLibAVPath) then
    begin
      mmoLog.Lines.Add(FFEncoder.LastErrMsg);
      Exit;
    end;
  end;

  if not OpenDialog1.Execute then
    // cancel open file
    Exit;

  // ensure reset FFEncoder
  FFEncoder.ClearTasks;
  ProgressBar1.Position := 0;

  // open input file selected
  LInFileName := OpenDialog1.FileName;

  // you should determine the out filename
  LOutFileName := GenerateOutputFileName(LInFileName, '_ffvcl.mp4');

  LOptions := '-i "' + LInFileName + '"';

  LOptions := LOptions + ' -c:a aac';
  LOptions := LOptions + ' -b:a 128k -ar 44100 -ac 2';
  LOptions := LOptions + ' -c:v mpeg4 -s 320x240 -b:v 320k';
  LOptions := LOptions + ' -y "' + LOutFileName + '"';

  // add a task
  LIndex := FFEncoder.AddTask(LOptions);
  if LIndex < 0 then
  begin
    mmoLog.Lines.Add('');
    mmoLog.Lines.Add('***Add task error: ' + FFEncoder.LastErrMsg);
    mmoLog.Lines.Add('');
    Exit;
  end;

  mmoLog.Lines.Add('');
  mmoLog.Lines.Add('***Ready to go!');
  mmoLog.Lines.Add('');

  btnStart.Enabled := True;
  btnStart.SetFocus;
end;

procedure TfrmSimpleConverter.btnStartClick(Sender: TObject);
begin
  // set status of buttons
  btnOpen.Enabled := False;
  btnStart.Enabled := False;
  btnStop.Enabled := True;
  btnPause.Enabled := True;
  btnResume.Enabled := False;
  btnStop.SetFocus;

  FFEncoder.PreviewVideo := chkPreview.Checked;
  FFEncoder.PreviewAudio := chkPreview.Checked;
  FFEncoder.FrameRateEmulation := chkPreview.Checked;

  // procedure Start(ATaskCount: Integer);
  // ATaskCount: >  0, perform tasks in asynchronous/non-blocking mode
  //             >  1, perform multiple tasks in the same time
  //             <= 0, perform tasks in synchronous/blocking mode
  FFEncoder.Start(1);
end;

procedure TfrmSimpleConverter.btnStopClick(Sender: TObject);
begin
  btnStop.Enabled := False;
  FFEncoder.Stop; // only works in asynchronous mode
end;

procedure TfrmSimpleConverter.btnPauseClick(Sender: TObject);
begin
  btnPause.Enabled := False;
  btnResume.Enabled := True;
  FFEncoder.Pause; // only works in asynchronous mode
end;

procedure TfrmSimpleConverter.btnResumeClick(Sender: TObject);
begin
  btnPause.Enabled := True;
  btnResume.Enabled := False;
  FFEncoder.Resume; // only works in asynchronous mode
end;

procedure TfrmSimpleConverter.btnWebSiteClick(Sender: TObject);
  function FromEXE: string;
  var
    S: string;
  begin
    S := ChangeFileExt(ExtractFileName(Application.ExeName), '');
    S := StringReplace(S, '[', '', [rfReplaceAll]);
    S := StringReplace(S, ']', '', [rfReplaceAll]);
    S := StringReplace(S, ' ', '_', [rfReplaceAll]);
    Result := '/?from=exe_' + S;
  end;
begin
  ShellExecute(Application.Handle, 'Open',   {Do not Localize}
    PChar(LowerCase(SWebSite + FromEXE)), '',
    PChar(ExtractFilePath(Application.ExeName)), 1);
end;

procedure TfrmSimpleConverter.cboLogLevelChange(Sender: TObject);
begin
  FFLogger.LogLevel := TLogLevel(cboLogLevel.ItemIndex);
end;

procedure TfrmSimpleConverter.chkPreviewClick(Sender: TObject);
begin
  FFEncoder.PreviewVideo := chkPreview.Checked;
  FFEncoder.PreviewAudio := chkPreview.Checked;
  FFEncoder.FrameRateEmulation := chkPreview.Checked;
end;

procedure TfrmSimpleConverter.FFEncoderProgress(Sender: TObject;
  AProgressInfo: PProgressInfo);
begin
  // OnProgress event handler
{
  PProgressInfo = ^TProgressInfo;
  TProgressInfo = record
    TaskIndex: Integer;     // index of converting tasks
    FileIndex: Integer;     // index of input files in the current task
    FrameNumber: Integer;   // current frame number
    FPS: Integer;           // video converting speed, frames per second, not valid when only audio output
    Quality: Single;        // quality
    BitRate: Single;        // bitrate
    CurrentSize: Int64;     // current output file size in bytes
    CurrentDuration: Int64; // current duration time in microsecond
    TotalDuration: Int64;   // total output duration time in microsecond
  end;
}
  with AProgressInfo^ do
  begin
    lblStatus.Caption := Format('Frame number: %d; FPS: %d; Size: %d; Time: %d',
        [FrameNumber, FPS, CurrentSize, CurrentDuration]);
    if TotalDuration > 0 then
      ProgressBar1.Position := CurrentDuration * 100 div TotalDuration;
  end;
end;

procedure TfrmSimpleConverter.FFEncoderTerminate(Sender: TObject;
  const ATerminateInfo: TTerminateInfo);
begin
  // OnTerminate event handler
{
  TTerminateInfo = record
    TaskIndex: Integer;     // index of converting tasks, (-1) means all tasks are terminated
    Finished: Boolean;      // True means converted success, False means converting broken
    Exception: Boolean;     // True means Exception occured, False please ignore
    ExceptionMsg: string;   // Exception message
  end;
}
  if ATerminateInfo.TaskIndex < 0 then
  begin
    // set status of buttons
    btnOpen.Enabled := True;
    btnStart.Enabled := False;
    btnStop.Enabled := False;
    btnPause.Enabled := False;
    btnResume.Enabled := False;
  end
  else if ATerminateInfo.Finished then
  begin // TaskIndex task converted success
    ProgressBar1.Position := 100;
  end;
  if ATerminateInfo.Exception then // Exception occured, show exception message
    Application.MessageBox(PChar(ATerminateInfo.ExceptionMsg), PChar(Application.Title), MB_OK + MB_ICONWARNING);
end;

procedure TfrmSimpleConverter.FFLoggerLog(Sender: TObject; AThreadID: Integer;
  ALogLevel: TLogLevel; const ALogMsg: string);
begin
  mmoLog.Lines.Add(ALogMsg);
end;

end.
