unit PlayerFrm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ComCtrls, ExtCtrls, ExtDlgs, Jpeg, FFBaseComponent,
  FFBasePlay, FFPlay, FFLoad, FFLog, SyncObjs;

type
  TfrmPlayer = class(TForm)
    lblDuration: TLabel;
    lblCurrentPTS: TLabel;
    lblAudioDevice: TLabel;
    lblRenderOn: TLabel;
    lblVideoStream: TLabel;
    lblAudioStream: TLabel;
    lblSubtitleStream: TLabel;
    lblAspectRatio: TLabel;
    lblSpeed: TLabel;
    lblEqualizer: TLabel;
    lblShowMode: TLabel;
    Panel1: TPanel;
    TrackBar1: TTrackBar;
    btnOpen: TButton;
    btnURL: TButton;
    btnPause: TButton;
    btnStep: TButton;
    btnStop: TButton;
    btnCapture: TButton;
    btnWebSite: TButton;
    chkDisableVideo: TCheckBox;
    chkDisableAudio: TCheckBox;
    chkExternalSubtitle: TCheckBox;
    chkOpenPaused: TCheckBox;
    chkWaterMark: TCheckBox;
    cboAudioDevice: TComboBox;
    cboScreens: TComboBox;
    cboVideo: TComboBox;
    cboAudio: TComboBox;
    cboSubtitle: TComboBox;
    cboAspectRatio: TComboBox;
    chkVerticalFlip: TCheckBox;
    chkFrameHook: TCheckBox;
    chkVideoHook: TCheckBox;
    trbSpeed: TTrackBar;
    btnSpeed: TButton;
    trbBrightness: TTrackBar;
    btnBrightness: TButton;
    trbSaturation: TTrackBar;
    btnSaturation: TButton;
    trbHue: TTrackBar;
    btnHue: TButton;
    chkAudioHook: TCheckBox;
    prbLeft: TProgressBar;
    prbRight: TProgressBar;
    chkMute: TCheckBox;
    trbAudioVolume: TTrackBar;
    cboShowMode: TComboBox;
    chkFullScreen: TCheckBox;
    chkStayOnTop: TCheckBox;
    chkMsgLog: TCheckBox;
    FFPlayer: TFFPlayer;
    FFLogger: TFFLogger;
    OpenDialog1: TOpenDialog;
    SavePictureDialog1: TSavePictureDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormDestroy(Sender: TObject);
    procedure btnOpenClick(Sender: TObject);
    procedure btnPauseClick(Sender: TObject);
    procedure btnStepClick(Sender: TObject);
    procedure btnStopClick(Sender: TObject);
    procedure btnCaptureClick(Sender: TObject);
    procedure btnWebSiteClick(Sender: TObject);
    procedure cboScreensChange(Sender: TObject);
    procedure cboVideoChange(Sender: TObject);
    procedure cboAudioChange(Sender: TObject);
    procedure cboSubtitleChange(Sender: TObject);
    procedure cboAspectRatioChange(Sender: TObject);
    procedure chkVerticalFlipClick(Sender: TObject);
    procedure chkFrameHookClick(Sender: TObject);
    procedure chkVideoHookClick(Sender: TObject);
    procedure trbSpeedChange(Sender: TObject);
    procedure btnSpeedClick(Sender: TObject);
    procedure trbBrightnessChange(Sender: TObject);
    procedure btnBrightnessClick(Sender: TObject);
    procedure trbSaturationChange(Sender: TObject);
    procedure btnSaturationClick(Sender: TObject);
    procedure trbHueChange(Sender: TObject);
    procedure btnHueClick(Sender: TObject);
    procedure chkAudioHookClick(Sender: TObject);
    procedure chkMuteClick(Sender: TObject);
    procedure trbAudioVolumeChange(Sender: TObject);
    procedure cboShowModeChange(Sender: TObject);
    procedure chkFullScreenClick(Sender: TObject);
    procedure chkStayOnTopClick(Sender: TObject);
    procedure chkMsgLogClick(Sender: TObject);
    procedure Panel1DblClick(Sender: TObject);
    procedure Panel1MouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure Panel1MouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure Panel1MouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure TrackBar1Change(Sender: TObject);
    procedure FFPlayerFileOpen(Sender: TObject; const ADuration: Int64;
      AFrameWidth, AFrameHeight: Integer; var AScreenWidth, AScreenHeight: Integer);
    procedure FFPlayerOpenFailed(Sender: TObject);
    procedure FFPlayerFrameHook(Sender: TObject; APicture: PAVPicture;
      APixFmt: TAVPixelFormat; AWidth, AHeight: Integer; const APTS: Int64);
    procedure FFPlayerVideoHook(Sender: TObject; ABitmap: TBitmap;
      const APTS: Int64; var AUpdate: Boolean);
    procedure FFPlayerAudioHook(Sender: TObject; const APTS: Int64; ASample: PByte;
      ASize, ASampleRate, AChannels: Integer);
    procedure FFPlayerPosition2(Sender: TObject; const APTS, ABytes: Int64);
    procedure FFPlayerState(Sender: TObject; AState: TPlayState);
    procedure FFLoggerLog(Sender: TObject; AThreadID: Integer;
      ALogLevel: TLogLevel; const ALogMsg: string);
  private
    { Private declarations }
    FDuration: Int64;
    FIsBytes: Boolean;
    FTrackChanging: Boolean;
    FScreenControl: TWinControl;
    FLock: TCriticalSection;
    FRect: TRect;
    URL: string;
    function GetScreenHandle: HWND;
  public
    { Public declarations }
    FBrightness: Single;
    FSaturation: Single;
    FHue: Single;
  end;

function IsMouseDown: Boolean;

var
  frmPlayer: TfrmPlayer;

implementation

{$R *.dfm}

uses
  FrameEffects, // adjust eq and hue
  libavfilter,  // for PPAVFilter
  AVFilterStubs, // for avfilter_get_by_name()
  MsgLogFrm,
  libavformat,
  ShellAPI,
  MyUtils,
  FFUtils;

const
  CLibAVPath = 'LibAV';
  CWaterMark = 'watermark.png';

  SAppTitle = 'Demo of FFPlayer %s';
  SCaption = 'Demo of FFPlayer %s - Delphi FFmpeg VCL Components';
  SWebSiteC = 'http://www.CCAVC.com';
  SWebSiteE = 'http://www.DelphiFFmpeg.com';


  CDialogOptions = [ofHideReadOnly, ofFileMustExist, ofEnableSizing];
  CPictureFiles = '*.BMP;*.GIF;*.JPEG;*.JPG;*.PNG;';
  CAudioFiles = '*.AAC;*.AC3;*.APE;*.DTS;*.FLAC;*.M4A;*.MKA;*.MP2;*.MP3;' +
      '*.MPA;*.MPC;*.OFR;*.OGG;*.RA;*.TTA;*.WAV;*.WMA;';
  CVideoFiles = '*.3GP;*.ASF;*.AVI;*.AVM;*.AVS;*.DAT;*.FLV;*.MKV;*.MOV;' +
      '*.MP4;*.MPEG;*.MPG;*.NSV;*.OGM;*.RM;*.RMVB;*.TP;*.TS;*.VOB;*.WMV;';
  CDialogFilter =
      'Video/Audio/Picture Files|' + CVideoFiles + CAudioFiles + CPictureFiles +
      '|Video Files|' + CVideoFiles +
      '|Audio Files|' + CAudioFiles +
      '|Picture Files|' + CPictureFiles +
      '|All Files|*.*';

  SHookTimeStamp = 'FFPlayer - Time Stamp: %d';

var
  SWebSite: string = SWebSiteE;

{$IF CompilerVersion <= 18.5} // Delphi 2007 and olders
type
  // Delphi 6 undefined
  // Delphi 7 to Delphi 2007 defined incorrectly, SizeOf(NativeInt) = 8, thus re-define it
  NativeInt = Integer;
  NativeUInt = Cardinal;
{$IFEND}

{$IF Defined(VER140) OR Defined(VER150)} // Delphi 6, 7
type
  TCriticalSectionEx = class(TCriticalSection)
  public
    function TryEnter: Boolean;
  end;
function TCriticalSectionEx.TryEnter: Boolean;
begin
  Result := TryEnterCriticalSection(FSection);
end;
{$IFEND}
// maximize or normal form
{$IF Defined(VER140) OR Defined(VER150) OR Defined(VER170)} // Delphi 6, 7, 2005
procedure GetBorderStyles(AForm: TCustomForm; ABorderStyle: TFormBorderStyle;
  var Style, ExStyle, ClassStyle: Cardinal);
begin
  // Clear existing border styles
  Style := Style and not (WS_POPUP or WS_CAPTION or WS_BORDER or WS_THICKFRAME or WS_DLGFRAME or DS_MODALFRAME);
  ExStyle := ExStyle and not (WS_EX_DLGMODALFRAME or WS_EX_WINDOWEDGE or WS_EX_TOOLWINDOW);
  ClassStyle := ClassStyle and not (CS_SAVEBITS or CS_BYTEALIGNWINDOW);

  // Set new border styles
  case ABorderStyle of
    bsNone:
      if (AForm.Parent = nil) and (AForm.ParentWindow = 0) then
        Style := Style or WS_POPUP;
    bsSingle, bsToolWindow:
      Style := Style or (WS_CAPTION or WS_BORDER);
    bsSizeable, bsSizeToolWin:
      Style := Style or (WS_CAPTION or WS_THICKFRAME);
    bsDialog:
      begin
        Style := Style or WS_POPUP or WS_CAPTION;
        ExStyle := ExStyle or WS_EX_DLGMODALFRAME or WS_EX_WINDOWEDGE;
        if not NewStyleControls then
          Style := Style or WS_DLGFRAME or DS_MODALFRAME;
        ClassStyle := ClassStyle or CS_DBLCLKS or CS_SAVEBITS or CS_BYTEALIGNWINDOW;
      end;
  end;
  if ABorderStyle in [bsToolWindow, bsSizeToolWin] then
    ExStyle := ExStyle or WS_EX_TOOLWINDOW;
end;
procedure GetBorderIconStyles(AForm: TCustomForm; ABorderStyle: TFormBorderStyle;
  var Style, ExStyle: Cardinal);
var
  LIcons: TBorderIcons;
begin
  // Clear existing border icon styles
  Style := Style and not (WS_MINIMIZEBOX or WS_MAXIMIZEBOX or WS_SYSMENU);
  ExStyle := ExStyle and not WS_EX_CONTEXTHELP;

  // Adjust icons based on border style
  LIcons := TForm(AForm).BorderIcons;
  case ABorderStyle of
    bsNone: LIcons := [];
    bsDialog: LIcons := LIcons * [biSystemMenu, biHelp];
    bsToolWindow,
    bsSizeToolWin: LIcons := LIcons * [biSystemMenu];
  end;

  // Set border icon styles
  if ABorderStyle in [bsSingle, bsSizeable, bsNone] then
  begin
    if biMinimize in LIcons then Style := Style or WS_MINIMIZEBOX;
    if biMaximize in LIcons then Style := Style or WS_MAXIMIZEBOX;
  end;
  if biSystemMenu in LIcons then Style := Style or WS_SYSMENU;
  if biHelp in LIcons then ExStyle := ExStyle or WS_EX_CONTEXTHELP;
end;
procedure SetBorderStyle(AForm: TCustomForm; ABorderStyle: TFormBorderStyle);
var
  LStyle, LExStyle, LClassStyle: Cardinal;
  LIcon: HICON;
begin
  LStyle := GetWindowLong(AForm.Handle, GWL_STYLE);
  LExStyle := GetWindowLong(AForm.Handle, GWL_EXSTYLE);
  LClassStyle := GetClassLong(AForm.Handle, GCL_STYLE);

  GetBorderStyles(AForm, ABorderStyle, LStyle, LExStyle, LClassStyle);
  GetBorderIconStyles(AForm, ABorderStyle, LStyle, LExStyle);

  SetWindowLong(AForm.Handle, GWL_STYLE, LStyle);
  SetWindowLong(AForm.Handle, GWL_EXSTYLE, LExStyle);
  SetClassLong(AForm.Handle, GCL_STYLE, LClassStyle);

  // Update icon on window frame
  if NewStyleControls then
    if ABorderStyle <> bsDialog then
    begin
      LIcon := TForm(AForm).Icon.Handle;
      if LIcon = 0 then LIcon := Application.Icon.Handle;
      if LIcon = 0 then LIcon := LoadIcon(0, IDI_APPLICATION);
      SendMessage(AForm.Handle, WM_SETICON, ICON_BIG, LIcon);
    end
    else
      SendMessage(AForm.Handle, WM_SETICON, ICON_BIG, 0);

  // Reset system menu based on new border style
  GetSystemMenu(AForm.Handle, True);
  AForm.Perform(WM_NCCREATE, 0, 0);

  SetWindowPos(AForm.Handle, 0, 0, 0, 0, 0, SWP_FRAMECHANGED or SWP_NOMOVE or
    SWP_NOZORDER or SWP_NOSIZE or SWP_NOACTIVATE);
  AForm.Invalidate;
end;
procedure SetStayOnTop(AHandle: HWND; AStayOnTop: Boolean);
const
  HWND_STYLE: array[Boolean] of HWND = (HWND_NOTOPMOST, HWND_TOPMOST);
begin
  SetWindowPos(AHandle, HWND_STYLE[AStayOnTop], 0, 0, 0, 0,
    SWP_NOMOVE or SWP_NOSIZE or SWP_NOACTIVATE or SWP_NOOWNERZORDER);
end;
{$IFEND}
procedure MaxOrNormalForm(AForm: TForm; AMax: Boolean);
const
{$J+}
  LastBounds: TRect = ();
{$J-}
begin
  if AMax then
  begin
    LastBounds := AForm.BoundsRect;
{$IF Defined(VER140) OR Defined(VER150) OR Defined(VER170)} // Delphi 6, 7, 2005
    SetStayOnTop(AForm.Handle, True);
    SetBorderStyle(AForm, bsNone);
{$ELSE}
    AForm.FormStyle := fsStayOnTop;
    AForm.BorderStyle := bsNone;
{$IFEND}
    AForm.BoundsRect := Screen.MonitorFromWindow(AForm.Handle).BoundsRect;
  end
  else
  begin
{$IF Defined(VER140) OR Defined(VER150) OR Defined(VER170)} // Delphi 6, 7, 2005
    SetStayOnTop(AForm.Handle, False);
    SetBorderStyle(AForm, bsSizeable);
{$ELSE}
    AForm.FormStyle := fsNormal;
    AForm.BorderStyle := bsSizeable;
{$IFEND}
    AForm.BoundsRect := LastBounds;
  end;
end;

// return desktop handle
function enumUserWindowsCB(ahwnd: HWND; lParam: LPARAM): BOOL; stdcall;
var
  wflags: Longint;
  sndWnd: HWND;
  targetWnd: HWND;
  resultHwnd: PLongWord;
begin
  wflags := GetWindowLong(ahwnd, GWL_STYLE);
  if (wflags and WS_VISIBLE) <> 0 then
  begin
    sndWnd := FindWindowEx(ahwnd, 0, 'SHELLDLL_DefView', nil);
    if sndWnd <> 0 then
    begin
      targetWnd := FindWindowEx(sndWnd, 0, 'SysListView32', 'FolderView');
      if targetWnd <> 0 then
      begin
        resultHwnd := PLongWord(lParam);
        resultHwnd^ := targetWnd;
        Result := False;
        Exit;
      end;
    end;
  end;
  Result := True;
end;
function GetDesktopHandle: HWND;
begin
  // works under Windows XP or classic theme under Windows Vista/7
  Result := FindWindow('ProgMan', nil);    {Do not Localize}
  if Result <> 0 then
  begin
    Result := GetWindow(Result, GW_CHILD);
    if Result <> 0 then
    begin
      Result := GetWindow(Result, GW_CHILD);
      if Result <> 0 then
        Exit;
    end;
  end;
  // works under Vista/7
  EnumWindows(@enumUserWindowsCB, NativeInt(@Result));
end;

// whether mouse is down
function IsMouseDown: Boolean;
begin
  if GetSystemMetrics(SM_SWAPBUTTON) <> 0 then
    Result := GetAsyncKeyState(VK_RBUTTON) and $8000 <> 0
  else
    Result := GetAsyncKeyState(VK_LBUTTON) and $8000 <> 0;
end;

// find wincontrol of form by name
function FindWinControl(AForm: TForm; const AName: string): TWinControl;
var
  I: Integer;
begin
  for I := 0 to AForm.ControlCount - 1 do
  begin
    if (AForm.Controls[I] is TWinControl) and (AForm.Controls[I].Name = AName) then
    begin
      Result := TWinControl(AForm.Controls[I]);
      Exit;
    end;
  end;
  Result := nil;
end;

type
  // audio level thread
  TLevelThread = class(TThread)
  private
    FEvent: TEvent;
    FLeftBar: TProgressBar;
    FRightBar: TProgressBar;
    FLeft: Integer;
    FRight: Integer;
  protected
    procedure Execute; override;
  public
    constructor Create(ALeftBar, ARightBar: TProgressBar);
    destructor Destroy; override;
    procedure Update(ALeft, ARight: Integer);
  end;

constructor TLevelThread.Create(ALeftBar, ARightBar: TProgressBar);
begin
  inherited Create(False);
  Self.FreeOnTerminate := True;
  FEvent := TEvent.Create(nil, True, False, '');
  FLeftBar := ALeftBar;
  FRightBar := ARightBar;
  FLeft := 0;
  FRight := 0;
end;

destructor TLevelThread.Destroy;
begin
  FEvent.Free;
end;

procedure TLevelThread.Execute;
begin
  while not Self.Terminated do
  begin
    FLeftBar.Position := Round(100 * FLeft / $8000); // $8000 -> 32768, max value of signed 16 bits
    FRightBar.Position := Round(100 * FRight / $8000);
    FEvent.ResetEvent;
    while FEvent.WaitFor(10) = wrTimeout do
      if Self.Terminated then
        Break;
  end;
end;

procedure TLevelThread.Update(ALeft, ARight: Integer);
begin
  FLeft := ALeft;
  FRight := ARight;
  FEvent.SetEvent;
end;

var
  FLevelThread: TLevelThread = nil;

procedure TfrmPlayer.FormCreate(Sender: TObject);
var
  I, T: Integer;
  Found: Boolean;
begin
  // initialize
  Application.Title := Format(SAppTitle, [FFPlayer.Version]);
  Self.Caption := Format(SCaption, [FFPlayer.Version]);

  if SysUtils.SysLocale.PriLangID = LANG_CHINESE then
    SWebSite := SWebSiteC
  else
    SWebSite := SWebSiteE;

{$IF CompilerVersion >= 35.0}
  TrackBar1.OnTracking := TrackBar1Change;
  trbSpeed.OnTracking := trbSpeedChange;
{$IFEND}

  btnWebsite.Hint := SWebSite;
  btnWebsite.ShowHint := True;
  AddMsgLog(SWebSite + sLineBreak);
  ShowMsgLogCheckBox := chkMsgLog;

  // ensure local decimal separator to be '.' and thousand separator to be ','
{$IF CompilerVersion >= 22.0}
  SysUtils.FormatSettings.DecimalSeparator := '.';
  SysUtils.FormatSettings.ThousandSeparator := ',';
{$ELSE}
  SysUtils.DecimalSeparator := '.';
  SysUtils.ThousandSeparator := ',';
{$IFEND}

  // equalizer
  FBrightness := 0;
  FSaturation := 1;
  FHue := 0;

  FFPlayer.AudioHook := chkAudioHook.Checked;

  // generate screen list
  cboScreens.Items.Clear;
  cboScreens.Items.BeginUpdate;
  try
    cboScreens.Items.Add('Desktop');
    cboScreens.Items.Add('Popup');
    T := 0;
    while True do
    begin
      Found := False;
      for I := 0 to Self.ControlCount - 1 do
      begin
        if (Self.Controls[I] is TWinControl) and ((Self.Controls[I] as TWinControl).TabOrder = T) then
        begin
          cboScreens.Items.Add(Self.Controls[I].Name);
          Inc(T);
          Found := True;
          Break;
        end;
      end;
      if not Found then
        Break;
    end;
    cboScreens.ItemIndex := 2;
  finally
    cboScreens.Items.EndUpdate;
  end;

{$IF Defined(VER140) OR Defined(VER150)} // Delphi 6, 7
  FLock := TCriticalSectionEx.Create;
{$ELSE}
  FLock := TCriticalSection.Create;
{$IFEND}
  FRect.Left := 0;
  FRect.Top := 0;
  FRect.Right := -1;
  FRect.Bottom := -1;

  // open dialog setting
  OpenDialog1.Options := CDialogOptions;
  OpenDialog1.Filter := CDialogFilter;

  // save dialog setting
  SavePictureDialog1.Options := [ofOverwritePrompt, ofHideReadOnly, ofExtensionDifferent, ofPathMustExist, ofEnableSizing];
  SavePictureDialog1.Filter := 'Bitmaps (*.bmp)|*.bmp|JPEG Image File |*.jpg;*.jpeg';
  SavePictureDialog1.DefaultExt := 'bmp';

  // disable all fpu exceptions(floating point exceptions):
  //   invalid operation, denormalized, divide by zero, overflow, underflow, inexact/precision
  FFPlayer.DisableFPUExceptions;
{
  // debug
  FFLogger.LogFile := 'F:\PlayDemo.log';
  FFLogger.LogToFile := True;
  FFLogger.ClearLogFile;
  FFLogger.Synchronous := True;
  FFLogger.OnLog := nil;
}
end;

procedure TfrmPlayer.FormKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #27 then
    chkFullScreen.Checked := False // exit full screen
  else if (Key = #32) and Panel1.Focused then
    FFPlayer.TogglePause;
end;

procedure TfrmPlayer.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  with FFPlayer do
  begin
    // Clear the event handlers
    OnPosition := nil;
    OnState := nil;
    OnFrameHook := nil;
    OnVideoHook := nil;
    OnAudioHook := nil;
  end;
  FFLogger.OnLog := nil;
  if Assigned(FLevelThread) then
  begin
    FLevelThread.Terminate;
    FLevelThread := nil;
  end;
end;

procedure TfrmPlayer.FormDestroy(Sender: TObject);
begin
  FLock.Free;
end;

function FindSubFile(APath, AFileName: string): string;
var
  LSearchRec: TsearchRec;
begin
  try
    if FindFirst(APath + AFileName, faAnyFile, LSearchRec) = 0 then
    begin
      repeat
        if ((LSearchRec.Attr and faAnyFile) <> 0) and
          (LSearchRec.Name <> '.') and (LSearchRec.Name <> '..') then
        begin
          Result := APath + LSearchRec.Name;
          Exit;
        end;
      until FindNext(LSearchRec) <> 0;
    end;
    Result := '';
  finally
    SysUtils.FindClose(LSearchRec);
  end;
end;

function CheckVideoFilter(AFilter: string): Boolean;
begin
  Assert(Assigned(avfilter_get_by_name));
  Result := Assigned(avfilter_get_by_name(PAnsiChar(AnsiString(AFilter))));
end;

function GetSubFile(AVideoFile: string): string;
const
  CExts: array[0..3] of string = ('srt', 'smi', 'ssa', 'ass');
var
  P, S, F: string;
  I: Integer;
begin
  P := ExtractFilePath(AVideoFile);
  S := ChangeFileExt(ExtractFileName(AVideoFile), '');
  // base file name subtitles
  for I := 0 to High(CExts) do
    if FileExists(P + S + '.' + CExts[I]) then
    begin
      Result := P + S + '.' + CExts[I];
      Exit;
    end;
  // base file name + * subtitles
  for I := 0 to High(CExts) do
  begin
    F := FindSubFile(P, S + '*.' + CExts[I]);
    if (F <> '') and FileExists(F) then
    begin
      Result := F;
      Exit;
    end;
  end;
  // * subtitles
{
  for I := 0 to High(CExts) do
  begin
    F := FindSubFile(P, '*.' + CExts[I]);
    if (F <> '') and FileExists(F) then
    begin
      Result := F;
      Exit;
    end;
  end;
}
  Result := '';
end;

{
function CheckFontConfig: Boolean;
var
  LTemp: array[0..255] of Char;
begin
  GetTempPath(255, LTemp);
  Result := FileExists(LTemp + 'fontconfig\cache\CACHEDIR.TAG');
end;
}

function TfrmPlayer.GetScreenHandle: HWND;
begin
  // wincontrol used for render video
  FScreenControl := FindWinControl(Self, cboScreens.Items.Strings[cboScreens.ItemIndex]);
  if Assigned(FScreenControl) then
    Result := FScreenControl.Handle
  else if cboScreens.Text = 'Desktop' then
    // render video on desktop
    Result := GetDesktopHandle
  else
    // render video on popup window
    Result := 0;
end;

procedure TfrmPlayer.btnOpenClick(Sender: TObject);
var
  LScreenHandle: HWND;
  LPngFile: string;
  LSubFile: string;
  LOptions: string;
begin
  // open media file/url to play

  // Load dynamic link libraries
  if not FFPlayer.AVLibLoaded then
  begin
    // TPathFileName = type WideString;
    // FFPlayer.LoadAVLib(const APath: TPathFileName): Boolean;
    // APath: Full path indicates location of FFmpeg DLLs.
    //        It can be empty, let Windows search DLLs in current dir or environment <PATH>
    //if not FFPlayer.LoadAVLib(ExtractFilePath(Application.ExeName) + CLibAVPath) then
    // the routine ExePath() is implemented in unit MyUtils which returns WideString type
    // of ExtractFilePath(Application.ExeName)
    if not FFPlayer.LoadAVLib(ExePath + CLibAVPath) then
    begin
      AddMsgLog(FFPlayer.LastErrMsg, True);
      Exit;
    end;

    if FFPlayer.InitSDL then
    begin
      // get audio devices after InitSDL()
      cboAudioDevice.Items.Assign(FFPlayer.AudioDevices);
      if cboAudioDevice.Items.Count > 1 then
        cboAudioDevice.Items.Insert(0, '<system default>');
      if cboAudioDevice.Items.Count > 0 then
        cboAudioDevice.ItemIndex := 0;
    end;
  end;

  if Sender = btnURL then
  begin
    // open url
    FFPlayer.ReadTimeout := 1000 * 60;
    if not InputQuery('Open', 'URL(or Filename)', URL) or (Trim(URL) = '') then
      // cancel open url
      Exit;
  end
  else
    // open media file
    if not OpenDialog1.Execute then
      // cancel open file
      Exit;

  // initial options
  LOptions := '';
  if chkDisableAudio.Checked then
    LOptions := LOptions + ' -an ';
  if chkDisableVideo.Checked then
    LOptions := LOptions + ' -vn ';
  if chkOpenPaused.Checked then
    LOptions := LOptions + ' -pause';

  // libavfilter: please refer to https://ffmpeg.org/ffmpeg-filters.html
  // NOTICE: What I know about libavfilter is also very limited.
  //         so you have to get help of it by yourself.

  //FFPlayer.VideoFilters := 'hue=H=2*PI*t:s=sin(2*PI*t)+1';
  FFPlayer.VideoFilters := Format('hue=b=%f:s=%f:H=%f', [FBrightness, FSaturation, FHue]);

  // text based external subtitles filter
  if chkExternalSubtitle.Checked and CheckVideoFilter('subtitles') and (Sender <> btnURL) then
  begin
    // fontconfig requires config file
    //if FileExists(ExtractFilePath(Application.ExeName) + 'fonts\fonts.conf') then
    begin
      LSubFile := GetSubFile(OpenDialog1.FileName);
      if LSubFile <> '' then
      begin
        AddMsgLog('Found external subtitle: ' + LSubFile);
        LSubFile := StringReplace(LSubFile, '\', '/', [rfReplaceAll]);
        LSubFile := StringReplace(LSubFile, ':', '\\:', [rfReplaceAll]);
        // default encoding is UTF-8, or special like XXX: subtitles=charenc=<XXX>:filename=<filename>
        FFPlayer.VideoFilters := FFPlayer.VideoFilters + ', subtitles=filename=' + LSubFile;
      end;
    end;
    //else
    //  AddMsgLog('Subtitles filter requires config file of FontConfig.', True);
  end
  else
    LSubFile := '';

  if chkWaterMark.Checked then
  begin
    LPngFile := ExtractFilePath(Application.ExeName) + CWaterMark;
    LPngFile := StringReplace(LPngFile, '\', '/', [rfReplaceAll]);
    LPngFile := StringReplace(LPngFile, ':', '\\:', [rfReplaceAll]);

    // other filters effect overlay
    //if FFPlayer.VideoFilters <> '' then
    //  FFPlayer.VideoFilters := ', ' + FFPlayer.VideoFilters;
    //FFPlayer.VideoFilters := Format('movie=%s [watermark]; [in] [watermark] overlay=main_w-overlay_w-10:10 %s [out]',
    //  [LPngFile, FFPlayer.VideoFilters]);

    // other filters do not effect overlay
    if FFPlayer.VideoFilters <> '' then
      FFPlayer.VideoFilters := FFPlayer.VideoFilters + ' [main]; [main] ';
    FFPlayer.VideoFilters := Format('movie=%s [watermark]; [in] %s [watermark] overlay=main_w-overlay_w-10:10 [out]',
      [LPngFile, FFPlayer.VideoFilters]);
{
    // two watermarks example
    FFPlayer.VideoFilters := Format('movie=%s [wm_left]; movie=%s [wm_right]; [in] %s [wm_left] overlay=10:10 [with_left];'+
      '[with_left] [wm_right] overlay=main_w-overlay_w-10:10 [out]',
      [LPngFile, LPngFile, FFPlayer.VideoFilters]);
}
  end;

  LScreenHandle := GetScreenHandle;

  // set audio device before Open()/TryOpen(), or leave it empty for default
  if cboAudioDevice.Text = '<system default>' then
    FFPlayer.AudioDevice := ''
  else
    FFPlayer.AudioDevice := cboAudioDevice.Text;

  if Sender = btnURL then
  begin
    // try to open and play url, render on the custom window specified by handle
    //if not FFPlayer.Open(URL, LScreenHandle, LOptions) then
    //  AddMsgLog(FFPlayer.LastErrMsg, True);
    FFPlayer.TryOpen(URL, LScreenHandle, LOptions);
  end
  else
    // try to open and play media file, render on the custom window specified by handle
    //if not FFPlayer.Open(OpenDialog1.FileName, LScreenHandle, LOptions) then
    //  AddMsgLog(FFPlayer.LastErrMsg, True);
    FFPlayer.TryOpen(OpenDialog1.FileName, LScreenHandle, LOptions);

  //if (LSubFile <> '') and not CheckFontConfig then
  //  Application.MessageBox('FontConfig might scan fonts folder. if so, just wait a moment and then re-open the video file.', 'Information', MB_ICONINFORMATION);
end;

procedure TfrmPlayer.btnPauseClick(Sender: TObject);
begin
  // toggle pause/resume
{
  if FFPlayer.Paused then
    FFPlayer.Resume
  else
    FFPlayer.Pause;
}
  // built-in toggle pause/resume
  FFPlayer.TogglePause;
end;

procedure TfrmPlayer.btnStepClick(Sender: TObject);
begin
  // step to next frame
  FFPlayer.StepToNextFrame;
end;

procedure TfrmPlayer.btnStopClick(Sender: TObject);
begin
  // stop
  FFPlayer.Stop(True);
end;

procedure TfrmPlayer.btnCaptureClick(Sender: TObject);
var
  BMP: TBitmap;
  PTS: Int64;
begin
  // capture video frame
  if FFPlayer.CurrentFrame(BMP, PTS) then
  begin
    //BMP.SaveToFile('F:\snapshot_' + IntToStr(PTS) + '.bmp');
    SavePictureDialog1.FileName := ChangeFileExt(FFPlayer.FileName, '_snapshot_') + IntToStr(PTS);
    if SavePictureDialog1.Execute then
    begin
      if SameText(ExtractFileExt(SavePictureDialog1.FileName), '.bmp') then
        BMP.SaveToFile(SavePictureDialog1.FileName)
      else
        with TJPEGImage.Create do
        try
          Assign(BMP);
          SaveToFile(SavePictureDialog1.FileName)
        finally
          Free;
        end;
    end;
  end;
end;

procedure TfrmPlayer.btnWebSiteClick(Sender: TObject);
  function FromEXE: string;
  var
    S: string;
  begin
    S := ChangeFileExt(ExtractFileName(Application.ExeName), '');
    S := StringReplace(S, '[', '', [rfReplaceAll]);
    S := StringReplace(S, ']', '', [rfReplaceAll]);
    S := StringReplace(S, ' ', '_', [rfReplaceAll]);
    Result := '/?from=exe_' + S;
  end;
begin
  // web site
  ShellExecute(Application.Handle, 'Open',   {Do not Localize}
    PChar(LowerCase(SWebSite + FromEXE)), '',
    PChar(ExtractFilePath(Application.ExeName)), 1);
end;

procedure TfrmPlayer.cboScreensChange(Sender: TObject);
begin
  FFPlayer.ScreenHandle := GetScreenHandle;
end;

procedure TfrmPlayer.cboVideoChange(Sender: TObject);
var
  S: string;
begin
  // change video stream channel
  S := cboVideo.Text;
  if Pos('#', S) > 0 then
  begin
    S := Copy(S, Pos('#', S) + 1, MaxInt);
    FFPlayer.VideoStreamIndex := StrToInt(S);
    if FIsBytes then
      FFPlayer.Seek(FFPlayer.CurrentPos, [sfByte])
    else
      FFPlayer.Seek(FFPlayer.CurrentPTS);
  end;
end;

procedure TfrmPlayer.cboAudioChange(Sender: TObject);
var
  S: string;
begin
  // change audio stream channel
  S := cboAudio.Text;
  if Pos('#', S) > 0 then
  begin
    S := Copy(S, Pos('#', S) + 1, MaxInt);
    FFPlayer.AudioStreamIndex := StrToInt(S);
    if FIsBytes then
      FFPlayer.Seek(FFPlayer.CurrentPos, [sfByte])
    else
      FFPlayer.Seek(FFPlayer.CurrentPTS);
  end;
end;

procedure TfrmPlayer.cboSubtitleChange(Sender: TObject);
var
  S: string;
begin
  // change subtitle stream channel
  S := cboSubtitle.Text;
  if Pos('#', S) > 0 then
  begin
    S := Copy(S, Pos('#', S) + 1, MaxInt);
    FFPlayer.SubtitleStreamIndex := StrToInt(S);
    if FIsBytes then
      FFPlayer.Seek(FFPlayer.CurrentPos, [sfByte])
    else
      FFPlayer.Seek(FFPlayer.CurrentPTS);
  end;
end;

procedure TfrmPlayer.cboAspectRatioChange(Sender: TObject);
begin
{
  = 0 -> keeping original
  < 0 -> scaling to fit screen
  > 0 -> custom, for example: 4/3, 16/9, 1.85, 2.35
}
  // change aspect ratio
  case cboAspectRatio.ItemIndex of
    1: FFPlayer.AspectRatio := -1;
    2: FFPlayer.AspectRatio := 4 / 3;
    3: FFPlayer.AspectRatio := 16 / 9;
    4: FFPlayer.AspectRatio := 1.85;
    5: FFPlayer.AspectRatio := 2.35;
  else
       FFPlayer.AspectRatio := 0;
  end;
end;

procedure TfrmPlayer.chkVerticalFlipClick(Sender: TObject);
begin
  // toggle enable/disable vertical flip
  FFPlayer.VerticalFlip := chkVerticalFlip.Checked;
end;

procedure TfrmPlayer.chkFrameHookClick(Sender: TObject);
begin
  // toggle enable/disable frame hook
  FFPlayer.FrameHook := chkFrameHook.Checked;
end;

procedure TfrmPlayer.chkVideoHookClick(Sender: TObject);
begin
  // toggle enable/disable video hook
  FFPlayer.VideoHook := chkVideoHook.Checked;
end;

procedure TfrmPlayer.trbSpeedChange(Sender: TObject);
begin
  // change playback speed
  if IsMouseDown then
    Exit;
  if trbSpeed.Position = 0 then
  begin
    FFPlayer.PlaybackSpeed := 1.0;
    FFPlayer.SyncType := stAudio;
    if FIsBytes then
      FFPlayer.Seek(FFPlayer.CurrentPos, [sfByte])
    else
      FFPlayer.Seek(FFPlayer.CurrentPTS);
    lblSpeed.Caption := 'Playback Speed 100%';
  end
  else
  begin
    if trbSpeed.Position > 0 then
      FFPlayer.PlaybackSpeed := trbSpeed.Position * 15/100 + 1
    else
      FFPlayer.PlaybackSpeed := trbSpeed.Position * 15/1600 + 1;
    FFPlayer.SyncType := stExternal;
    lblSpeed.Caption := Format('Playback Speed %d%%', [Round(FFPlayer.PlaybackSpeed * 100)]);
  end;
end;

procedure TfrmPlayer.btnSpeedClick(Sender: TObject);
begin
  // reset playback speed
  trbSpeed.Position := 0;
end;

procedure TfrmPlayer.trbBrightnessChange(Sender: TObject);
begin
  // change brightness range (-10 - 10)
  FBrightness := trbBrightness.Position / 10;
  FFPlayer.SendVideoFilterCommand('hue', 'b', FloatToStr(FBrightness), 0);
end;

procedure TfrmPlayer.btnBrightnessClick(Sender: TObject);
begin
  // reset brightness
  trbBrightness.Position := 0;
end;

procedure TfrmPlayer.trbSaturationChange(Sender: TObject);
begin
  // change saturation range (-10 - 10)
  FSaturation := trbSaturation.Position / 10;
  FFPlayer.SendVideoFilterCommand('hue', 's', FloatToStr(FSaturation), 0);
end;

procedure TfrmPlayer.btnSaturationClick(Sender: TObject);
begin
  // reset saturation
  trbSaturation.Position := 10;
end;

procedure TfrmPlayer.trbHueChange(Sender: TObject);
begin
  // change hue degrees range (-PI - PI)
  FHue := trbHue.Position / 100 * PI;
  FFPlayer.SendVideoFilterCommand('hue', 'H', FloatToStr(FHue), 0);
end;

procedure TfrmPlayer.btnHueClick(Sender: TObject);
begin
  // reset hue
  trbHue.Position := 0;
end;

procedure TfrmPlayer.chkAudioHookClick(Sender: TObject);
begin
  // toggle enable/disable audio hook
  FFPlayer.AudioHook := chkAudioHook.Checked;
  prbLeft.Position := 0;
  prbRight.Position := 0;
end;

procedure TfrmPlayer.chkMuteClick(Sender: TObject);
begin
  // toggle audio mute
  FFPlayer.Mute := chkMute.Checked;
end;

procedure TfrmPlayer.trbAudioVolumeChange(Sender: TObject);
begin
  // change audio volume range (0-128)
  FFPlayer.AudioVolume := trbAudioVolume.Max - trbAudioVolume.Position;
end;

procedure TfrmPlayer.cboShowModeChange(Sender: TObject);
begin
  // change show mode
  FFPlayer.ShowMode := TShowMode(cboShowMode.ItemIndex);
end;

procedure TfrmPlayer.chkFullScreenClick(Sender: TObject);
const
  LAlign: array[Boolean] of TAlign = (alNone, alClient);
{$J+}
  LastBounds: TRect = ();
{$J-}
var
  LToFullScreen: Boolean;
  I: Integer;
begin
  // toggle full/normal screen
  LToFullScreen := Panel1.Align <> alClient;

  if LToFullScreen <> chkFullScreen.Checked then
    Exit;

  // save normal position and size
  if LToFullScreen then
    LastBounds := Panel1.BoundsRect;

  // toggle other controls visible
  for I := 0 to ControlCount - 1 do
    if Controls[I].Name <> 'Panel1' then
      TControl(Controls[I]).Visible := not LToFullScreen;

  // toggle full or normal screen
  MaxOrNormalForm(Self, LToFullScreen);
  Panel1.Align := LAlign[LToFullScreen];
  Sleep(10);

  // restore normal position and size
  if not LToFullScreen then
  begin
    Panel1.BoundsRect := LastBounds;
    Panel1.Anchors := [akLeft, akTop, akRight, akBottom];
  end;
end;

procedure TfrmPlayer.chkStayOnTopClick(Sender: TObject);
begin
  // toggle stay on top
  if chkStayOnTop.Checked and not FFPlayer.Paused then
{$IF Defined(VER140) OR Defined(VER150) OR Defined(VER170)} // Delphi 6, 7, 2005
    SetStayOnTop(Self.Handle, True)
  else
    SetStayOnTop(Self.Handle, False);
{$ELSE}
    Self.FormStyle := fsStayOnTop
  else
    Self.FormStyle := fsNormal;
{$IFEND}
end;

procedure TfrmPlayer.chkMsgLogClick(Sender: TObject);
begin
  // toggle message log
  ShowMsgLog(chkMsgLog.Checked);
end;

procedure TfrmPlayer.Panel1DblClick(Sender: TObject);
begin
  // toggle full/normal screen
  chkFullScreen.Checked := not chkFullScreen.Checked;
end;

procedure TfrmPlayer.Panel1MouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  // init FrameHook rect
  FLock.Enter;
  try
    FRect.Left := X;
    FRect.Top := Y;
    FRect.Right := X;
    FRect.Bottom := Y;
  finally
    FLock.Leave;
  end;
end;

procedure TfrmPlayer.Panel1MouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
begin
  // resize FrameHook rect
  if Shift = [ssLeft] then
{$IF Defined(VER140) OR Defined(VER150)} // Delphi 6, 7
    if TCriticalSectionEx(FLock).TryEnter then
{$ELSE}
    if FLock.TryEnter then
{$IFEND}
      try
        FRect.Right := X;
        FRect.Bottom := Y;
      finally
        FLock.Leave;
      end;
end;

procedure TfrmPlayer.Panel1MouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  // resize FrameHook rect
  FLock.Enter;
  try
    FRect.Right := X;
    FRect.Bottom := Y;
  finally
    FLock.Leave;
  end;
end;

procedure TfrmPlayer.TrackBar1Change(Sender: TObject);
begin
  // do seek when change track bar
  if not FTrackChanging and not IsMouseDown and (FDuration > 0) then
    if FIsBytes then
      FFPlayer.Seek(FDuration * TrackBar1.Position div TrackBar1.Max, [sfByte])
    else
      FFPlayer.Seek(FDuration * TrackBar1.Position div TrackBar1.Max);
end;

function BytesToStr(AFileSize: Int64): string;
begin
  Result := Format('%.3d', [AFileSize mod 1000]);
  AFileSize := Trunc(AFileSize / 1000);
  while AFileSize > 0 do
  begin
    Result := Format('%.3d,%s', [AFileSize mod 1000, Result]);
    AFileSize := Trunc(AFileSize / 1000);
  end;
end;

function DurationToStr(ADuration: Int64): string;
var
  hours, mins, secs, us: Integer;
begin
  if ADuration <> AV_NOPTS_VALUE then
  begin
    secs := (ADuration + 5000) div AV_TIME_BASE;
    us := (ADuration + 5000) mod AV_TIME_BASE;
    mins := secs div 60;
    secs := secs mod 60;
    hours := mins div 60;
    mins := mins mod 60;
    Result := Format('%.2d:%.2d:%.2d.%.3d',
                    [hours, mins, secs, (1000 * us) div AV_TIME_BASE]);
  end
  else
    Result := 'N/A';
end;

procedure TfrmPlayer.FFPlayerFileOpen(Sender: TObject; const ADuration: Int64;
  AFrameWidth, AFrameHeight: Integer; var AScreenWidth, AScreenHeight: Integer);
var
  I: Integer;
  aspect_ratio: Single;
  P: PAVStream;
begin
  // OnFileOpen event handler

  // show media file info
  AddMsgLog(FFPlayer.Decoder.FileInfoText);

  // aspect ratio
  if FFPlayer.VideoStreamIndex >= 0 then
  begin
    P := PPtrIdx(FFPlayer.Decoder.FileHandle.streams, FFPlayer.VideoStreamIndex);
    with P.codecpar^ do
    begin
      if (sample_aspect_ratio.num <> 0) and (sample_aspect_ratio.den <> 0) then
        aspect_ratio := sample_aspect_ratio.num / sample_aspect_ratio.den
      else
        aspect_ratio := 1.0;
      aspect_ratio := aspect_ratio * width / height;
    end;
    AddMsgLog(Format('aspect ratio: %f', [aspect_ratio]));
  end;

  // show information
  AddMsgLog(Format('file size: %s, duration: %s, frame size: %dx%d',
    [BytesToStr(FFPlayer.Decoder.FileSize), DurationToStr(ADuration), AFrameWidth, AFrameHeight]));

  // tell player the screen size
  if Assigned(FScreenControl) then
  begin
    // win control
    AScreenWidth := FScreenControl.Width;
    AScreenHeight := FScreenControl.Height;
  end
  else if cboScreens.Text = 'Desktop' then
  begin
    // desktop
    AScreenWidth := Screen.DesktopWidth;
    AScreenHeight := Screen.DesktopHeight;
  end
  else
  begin
    // popup window
    AScreenWidth := 640; // Screen.Width;
    AScreenHeight := 480; // Screen.Height;
  end;

  // duration of media file
  FDuration := ADuration;
  if (FDuration < 0) and (FFPlayer.Decoder.FileSize > 0) then
  begin
    FIsBytes := True;
    FDuration := FFPlayer.Decoder.FileSize;
  end
  else
    FIsBytes := False;
  if FIsBytes then
    lblDuration.Caption := BytesToStr(FDuration)
  else
    lblDuration.Caption := DurationToStr(ADuration);
  lblCurrentPTS.Caption := 'N/A';

  // setup track bar
  TrackBar1.Frequency := 5;
  TrackBar1.TickStyle := tsAuto;
  TrackBar1.Max := TrackBar1.Width;
  TrackBar1.SelStart := 0;
  TrackBar1.SelEnd := 0;
  TrackBar1.SliderVisible := FDuration > 0;

  FTrackChanging := True;
  try
    TrackBar1.Position := 0;
  finally
    FTrackChanging := False;
  end;

  // setup stream info, including video, audio and subtitle
  cboVideo.Items.BeginUpdate;
  cboAudio.Items.BeginUpdate;
  cboSubtitle.Items.BeginUpdate;
  try
    cboVideo.Items.Clear;
    cboAudio.Items.Clear;
    cboSubtitle.Items.Clear;
    for I := 0 to FFPlayer.Decoder.StreamCount - 1 do
    begin
      if FFPlayer.Decoder.IsVideoStream(I) then
        cboVideo.Items.Add(Format('Video#%d', [I]))
      else if FFPlayer.Decoder.IsAudioStream(I) then
        cboAudio.Items.Add(Format('Audio#%d', [I]))
      else if FFPlayer.Decoder.IsSubtitleStream(I) then //and (FFPlayer.Decoder.SubtitleStreamInfos[I].SubFormat = 0) then
        cboSubtitle.Items.Add(Format('Subtitle#%d', [I]));
    end;
    cboVideo.Enabled := (cboVideo.Items.Count > 1) and (FFPlayer.VideoStreamIndex >= 0);
    cboVideo.ItemIndex := cboVideo.Items.IndexOf(Format('Video#%d', [FFPlayer.VideoStreamIndex]));
    cboAudio.Enabled := (cboAudio.Items.Count > 1) and (FFPlayer.AudioStreamIndex >= 0);
    cboAudio.ItemIndex := cboAudio.Items.IndexOf(Format('Audio#%d', [FFPlayer.AudioStreamIndex]));
    cboSubtitle.Enabled := (cboSubtitle.Items.Count > 0) and (FFPlayer.VideoStreamIndex >= 0);
    if cboSubtitle.Enabled then
    begin
      cboSubtitle.Items.Insert(0, 'Subtitle#-1');
      cboSubtitle.ItemIndex := cboSubtitle.Items.IndexOf(Format('Subtitle#%d', [FFPlayer.SubtitleStreamIndex]));
    end;
  finally
    cboVideo.Items.EndUpdate;
    cboAudio.Items.EndUpdate;
    cboSubtitle.Items.EndUpdate;
  end;
end;

procedure TfrmPlayer.FFPlayerOpenFailed(Sender: TObject);
begin
  AddMsgLog(FFPlayer.LastErrMsg, True);
end;

procedure TfrmPlayer.FFPlayerFrameHook(Sender: TObject; APicture: PAVPicture;
  APixFmt: TAVPixelFormat; AWidth, AHeight: Integer; const APTS: Int64);
var
  R: TRect;
begin
  // OnFrameHook event handler

  // example for YUV420P <- APixFmt
// adjust rect frame -> (x1, y1) - (x2, y2)
// adjust_eq(PByte(NativeUInt(data[0]) + linesize[0] * y1  + x1),
//           linesize[0],
//           x2 - x1,
//           y2 - y1,
//           brightness,
//           contrast);
{
  FrameEffects.adjust_eq(PByte(NativeUInt(APicture.data[0]) + APicture.linesize[0] * y1 + x1),
           APicture.linesize[0],
           x2 - x1,
           y2 - y1,
           20,
           0);
}
// adjust_hue(PByte(NativeUInt(data[1]) + linesize[1] * (y1 shr vsub) + (x1 shr vsub)),
//            PByte(NativeUInt(data[2]) + linesize[2] * (y1 shr vsub) + (x1 shr vsub)),
//            linesize[1],
//            (x2 - x1) shr hsub,
//            (y2 - y1) shr vsub,
//            hue,
//            saturation);
{
 FrameEffects.adjust_hue(PByte(NativeUInt(APicture.data[1]) + APicture.linesize[1] * (y1 shr 1) + (x1 shr 1)),
            PByte(NativeUInt(APicture.data[2]) + APicture.linesize[2] * (y1 shr 1) + (x1 shr 1)),
            APicture.linesize[1],
            (x2 - x1) shr 1,
            (y2 - y1) shr 1,
            0,
            -100);
}
  FLock.Enter;
  try
    R := FFPlayer.DisplayToActualRect(FRect);
    if (R.Right > R.Left) and (R.Bottom > R.Top) then
      FrameEffects.adjust_eq(PByte(NativeUInt(APicture.data[0]) + NativeUInt(APicture.linesize[0] * R.Top + R.Left)),
               APicture.linesize[0],
               R.Right - R.Left,
               R.Bottom - R.Top,
               20,
               0);
  finally
    FLock.Leave;
  end;
end;

procedure TfrmPlayer.FFPlayerVideoHook(Sender: TObject; ABitmap: TBitmap;
  const APTS: Int64; var AUpdate: Boolean);
begin
  // OnVideoHook event handler

  with ABitmap.Canvas.Font do
  begin // setup font example
    Color := clWhite;
    Name := 'Tahoma';
    Size := 12;
    Style := [fsBold, fsUnderline];
  end;
  // text out with Presentation Time Stamp example
  ABitmap.Canvas.TextOut(10, 10, Format(SHookTimeStamp, [APTS]));
  // draw graphic example
  ABitmap.Canvas.Draw(ABitmap.Width - Application.Icon.Width - 10,
                      ABitmap.Height - Application.Icon.Height - 10,
                      Application.Icon);
end;

procedure GetLevelS16(ASample: PByte; ASize, ASampleRate, AChannels: Integer;
  out L, R: Integer);
var
  P: PSmallInt;
  I, C: Integer;
  L1, L2: SmallInt;
begin
  // get level of audio sample in signed 16 bits format
  P := PSmallInt(ASample);
  L1 := 0;
  L2 := 0;
  for I := 0 to ASize div SizeOf(SmallInt) div AChannels - 1 do
  begin
    for C := 0 to AChannels - 1 do
    begin
      if (C mod 2) = 0 then
      begin
        if Abs(P^) > L1 then
          L1 := Abs(P^) and $7FFF;
      end
      else
      begin
        if Abs(P^) > L2 then
          L2 := Abs(P^) and $7FFF;
      end;
      Inc(P);
    end;
  end;
  L := L1;
  R := L2;
end;

procedure TfrmPlayer.FFPlayerAudioHook(Sender: TObject; const APTS: Int64;
  ASample: PByte; ASize, ASampleRate, AChannels: Integer);
var
  L1, L2: Integer;
//  P: PSmallInt;
//  I, C: Integer;
begin
  // OnAudioHook event handler

  // NOTICE: sample format is always AV_SAMPLE_FMT_S16 (signed 16 bits)

  // example for showing audio sample level (peak amplitude)
  GetLevelS16(ASample, ASize, ASampleRate, AChannels, L1, L2);
  //prbLeft.Position := Round(100 * L1 / $8000); // $8000 -> 32768, max value of signed 16 bits
  //prbRight.Position := Round(100 * L2 / $8000);
  if not Assigned(FLevelThread) then
    FLevelThread := TLevelThread.Create(prbLeft, prbRight);
  FLevelThread.Update(L1, L2);

  // example for disabling left channel or right channel, toggle every 5 seconds
{
  P := PSmallInt(ASample);
  for I := 0 to ASize div SizeOf(SmallInt) div AChannels - 1 do
  begin
    if APTS div 5000000 mod 2 = 0 then
    begin
      for C := 0 to AChannels - 1 do
      begin
        if (C mod 2) = 0 then
          P^ := 0;    // disable left channel
          //P^ := PSmallInt(PAnsiChar(P) + SizeOf(SmallInt))^;  // copy right channel to left channel
        Inc(P);
      end;
    end
    else
    begin
      for C := 0 to AChannels - 1 do
      begin
        if (C mod 2) = 1 then
          P^ := 0;    // disable right channel
          //P^ := PSmallInt(PAnsiChar(P) - SizeOf(SmallInt))^;  // copy left channel to right channel
        Inc(P);
      end;
    end;
  end;
  if APTS div 5000000 mod 2 = 0 then
  begin
    for C := 0 to ASize div SizeOf(SmallInt) - ASize div SizeOf(SmallInt) div AChannels * AChannels - 1 do
    begin
      if (C mod 2) = 0 then
        P^ := 0; // disable left channel
        //P^ := PSmallInt(PAnsiChar(P) + SizeOf(SmallInt))^;  // copy right channel to left channel
      Inc(P);
    end;
  end
  else
  begin
    for C := 0 to ASize div SizeOf(SmallInt) - ASize div SizeOf(SmallInt) div AChannels * AChannels - 1 do
    begin
      if (C mod 2) = 1 then
        P^ := 0; // disable right channel
        //P^ := PSmallInt(PAnsiChar(P) - SizeOf(SmallInt))^;  // copy left channel to right channel
      Inc(P);
    end;
  end;
}

  // example for enabling only special channel
{
  if FEnableOnlyChannel > 0 then
  begin
    P := PSmallInt(ASample);
    for I := 0 to ASize div SizeOf(SmallInt) div AChannels - 1 do
    begin
      for J := 1 to AChannels do
      begin
        if FEnableOnlyChannel <> J then
          P^ := 0;
        Inc(P);
      end;
    end;
    for J := 1 to ASize div SizeOf(SmallInt) - ASize div SizeOf(SmallInt) div AChannels * AChannels do
    begin
      if FEnableOnlyChannel <> J then
        P^ := 0;
      Inc(P);
    end;
  end;
}
end;

procedure TfrmPlayer.FFPlayerPosition2(Sender: TObject; const APTS, ABytes: Int64);
begin
  // OnPosition event handler
  if FIsBytes then
    lblCurrentPTS.Caption := BytesToStr(ABytes)
  else
    lblCurrentPTS.Caption := DurationToStr(APTS);
  if ((APTS >= 0) or (ABytes >= 0)) and (FDuration > 0) then
  begin
    if FIsBytes then
      TrackBar1.SelEnd := TrackBar1.Max * ABytes div FDuration
    else
      TrackBar1.SelEnd := TrackBar1.Max * APTS div FDuration;
    if not FTrackChanging and (not IsMouseDown or not TrackBar1.Focused) then
    begin
      FTrackChanging := True;
      try
        if FIsBytes then
          TrackBar1.Position := TrackBar1.Max * ABytes div FDuration
        else
          TrackBar1.Position := TrackBar1.Max * APTS div FDuration;
      finally
        FTrackChanging := False;
      end;
    end;
  end;
end;

procedure TfrmPlayer.FFPlayerState(Sender: TObject; AState: TPlayState);
const
  CPlayState: array[TPlayState] of string = ('Play', 'Pause', 'Resume', 'Step', 'Stop', 'End');
begin
  // OnState event handler

  // show state
  AddMsgLog(CPlayState[AState]);

  case AState of
    psPlay, psResume:
      if chkStayOnTop.Checked then
{$IF Defined(VER140) OR Defined(VER150) OR Defined(VER170)} // Delphi 6, 7, 2005
        SetStayOnTop(Self.Handle, True);
{$ELSE}
        Self.FormStyle := fsStayOnTop;
{$IFEND}
    psPause, psStep:
{$IF Defined(VER140) OR Defined(VER150) OR Defined(VER170)} // Delphi 6, 7, 2005
      SetStayOnTop(Self.Handle, False);
{$ELSE}
      Self.FormStyle := fsNormal;
{$IFEND}
    psStop:
      begin
{$IF Defined(VER140) OR Defined(VER150) OR Defined(VER170)} // Delphi 6, 7, 2005
        SetStayOnTop(Self.Handle, False);
{$ELSE}
        Self.FormStyle := fsNormal;
{$IFEND}
        // repaint the screen to the original appearance
        if HWND(FFPlayer.ScreenHandle) <> 0 then
        begin
          PostMessage(FFPlayer.ScreenHandle, CM_INVALIDATE, 0, 0);
          UpdateWindow(FFPlayer.ScreenHandle);
        end;
      end;
    psEnd:
      begin
//        FFPlayer.Stop;
        FFPlayer.Pause;
        if FIsBytes then
          FFPlayer.Seek(0, [sfByte])
        else
          FFPlayer.Seek(0, [sfBackward]);
      end;
  end;

  // loop example
{
  if AState = psEnd then
    if FIsBytes then
      FFPlayer.Seek(0, [sfByte])
    else
      FFPlayer.Seek(0, [sfBackward]);
}

  // play next example, need a Timer
{
  if AState = psEnd then
  begin
    Timer1.Interval := 20;
    Timer1.Enabled := True;
    // in Timer1.OnTimer
    //Timer1.Enabled := False;
    //FFPlayer.TryOpen('<the next file>', TheScreenHandle);
  end;
}
end;

procedure TfrmPlayer.FFLoggerLog(Sender: TObject; AThreadID: Integer;
  ALogLevel: TLogLevel; const ALogMsg: string);
begin
  AddMsgLog(ALogMsg, ALogLevel in [llPanic, llFatal, llError]);
end;

end.
