unit OutputFormatFrm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls;

type
  TfrmOutputFormat = class(TForm)
    lblFormat: TLabel;
    lblFormatLong: TLabel;
    lblExtension: TLabel;
    lblVideoCodec: TLabel;
    lblVideoCodecLong: TLabel;
    lblAudioCodec: TLabel;
    lblAudioCodecLong: TLabel;
    cboExtension: TComboBox;
    txtFormat: TEdit;
    txtFormatLong: TEdit;
    txtVideoCodec: TEdit;
    txtVideoCodecLong: TEdit;
    txtAudioCodec: TEdit;
    txtAudioCodecLong: TEdit;
    mmoLog: TMemo;
    procedure FormCreate(Sender: TObject);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure cboExtensionChange(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  frmOutputFormat: TfrmOutputFormat;

implementation

{$R *.dfm}

uses
  FFLoad,
  libavcodec,
  libavcodec_codec,
  libavcodec_codec_id,
  AVCodecStubs,
  libavformat,
  AVFormatStubs,
  libavutil;

const
  CLibAVPath = 'LibAV';

// enumerate all output format
procedure GetAllOutputFormatExtensions(AList: TStrings);
var
  P: PAVOutputFormat;
  I: Integer;
  S, E: string;
begin
  Assert(Assigned(av_muxer_iterate));
  I := 0;
  P := av_muxer_iterate(@I);
  while Assigned(P) do
  begin
    S := LowerCase(string(P.extensions));
    while Pos(',', S) > 1 do
    begin
      E := Trim(Copy(S, 1, Pos(',', S) - 1));
      if E <> '' then
        AList.Add(E);
      S := Trim(Copy(S, Pos(',', S) + 1, MaxInt));
    end;
    if S <> '' then
      AList.Add(S);
    P := av_muxer_iterate(@I);
  end;
end;

// guess output format, return codec name
type
  TFindType = (ftFormat, ftVideo, ftAudio);
function FindOutputFormatCodec(AOutputFileName: string; AFindType: TFindType;
  var AName, ANameLong, AErrMsg: string): Boolean;
const
  CAVMediaType: array[TFindType] of TAVMediaType =
    (AVMEDIA_TYPE_UNKNOWN, AVMEDIA_TYPE_VIDEO, AVMEDIA_TYPE_AUDIO);
  CAudioVideo: array[TFindType] of string = ('', 'video', 'audio');
var
  F: PAVOutputFormat;
  LCodecID: TAVCodecID;
  LAVCodec: PAVCodec;
begin
  Assert(Assigned(av_guess_format) and Assigned(av_guess_codec) and Assigned(avcodec_find_encoder));

  Result := False;

  F := av_guess_format(nil, PAnsiChar(AnsiString(AOutputFileName)), nil);
  if not Assigned(F) then
  begin
    AErrMsg := 'Unable to find a suitable output format for ' + ExtractFileExt(AOutputFileName);
    Exit;
  end;

  if AFindType = ftFormat then
  begin
    AName := string(F.name);
    ANameLong := string(F.long_name);
    Result := True;
    Exit;
  end;

  //(LCodecID = F.video_codec) or (LCodecID = F.audio_codec)
  LCodecID := av_guess_codec(F, nil, PAnsiChar(AnsiString(AOutputFileName)), nil, CAVMediaType[AFindType]);
  if LCodecID = AV_CODEC_ID_NONE then
  begin
    AErrMsg := Format('No %s stream for %s', [CAudioVideo[AFindType], ExtractFileExt(AOutputFileName)]);
    Exit;
  end;

  LAVCodec := avcodec_find_encoder(LCodecID);
  if not Assigned(LAVCodec) then
  begin
    AErrMsg := Format('Unable to find %s codec(encoder) for %s', [CAudioVideo[AFindType], ExtractFileExt(AOutputFileName)]);
    Exit;
  end;

  AName := string(LAVCodec.name);
  ANameLong := string(LAVCodec.long_name);
  Result := True;
end;

procedure TfrmOutputFormat.FormCreate(Sender: TObject);
var
  SL: TStringList;
begin
  // Load dynamic link libraries
  FFLoader.LibraryPath := ExtractFilePath(Application.ExeName) + CLibAVPath;
  if not FFLoader.Load([avCodec, avFormat]) then
  begin
    mmoLog.Lines.Add(FFLoader.LastErrMsg);
    Exit;
  end;

  SL := TStringList.Create;
  try
    SL.BeginUpdate;
    try
      SL.Sorted := True;
      GetAllOutputFormatExtensions(SL);
      SL.Sort;
    finally
      SL.EndUpdate;
    end;
    cboExtension.Items.Assign(SL);
    cboExtension.Text := '';
    cboExtension.Enabled := True;
  finally
    SL.Free;
  end;
  mmoLog.Lines.Add('Please input/select the extension of the output filename');
end;

procedure TfrmOutputFormat.FormKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #27 then
    Close;
end;

procedure TfrmOutputFormat.cboExtensionChange(Sender: TObject);
var
  S: string;
  LName: string;
  LNameLong: string;
  LErrMsg: string;
begin
  S := cboExtension.Text;
  txtFormat.Text := '';
  txtFormatLong.Text := '';
  txtVideoCodec.Text := '';
  txtVideoCodecLong.Text := '';
  txtAudioCodec.Text := '';
  txtAudioCodecLong.Text := '';
  if Trim(S) = '' then
    Exit;
  S := 'aaa.' + LowerCase(S);

  // format
  if FindOutputFormatCodec(S, ftFormat, LName, LNameLong, LErrMsg) then
  begin
    txtFormat.Text := LName;
    txtFormatLong.Text := LNameLong;

    // video
    if FindOutputFormatCodec(S, ftVideo, LName, LNameLong, LErrMsg) then
    begin
      txtVideoCodec.Text := LName;
      txtVideoCodecLong.Text := LNameLong;
    end
    else
      mmoLog.Lines.Add(LErrMsg);

    // audio
    if FindOutputFormatCodec(S, ftAudio, LName, LNameLong, LErrMsg) then
    begin
      txtAudioCodec.Text := LName;
      txtAudioCodecLong.Text := LNameLong;
    end
    else
      mmoLog.Lines.Add(LErrMsg);
  end
  else
    mmoLog.Lines.Add(LErrMsg);
end;

end.
