unit FFmpegCommandFrm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ComCtrls, FFBaseComponent, FFEncode, FFLoad, FFLog,
  FFDecode;

type
  TfrmFFmpegCommand = class(TForm)
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    lblStatus: TLabel;
    btnStart: TButton;
    btnStop: TButton;
    btnPause: TButton;
    btnResume: TButton;
    btnWebSite: TButton;
    cboLogLevel: TComboBox;
    chkPreview: TCheckBox;
    cboInputOptions: TComboBox;
    cboInputFile: TComboBox;
    btnInputFile: TButton;
    btnFileInfo: TButton;
    cboOutputOptions: TComboBox;
    cboOutputFile: TComboBox;
    btnOutputFile: TButton;
    mmoLog: TMemo;
    ProgressBar1: TProgressBar;
    FFEncoder: TFFEncoder;
    FFLogger: TFFLogger;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    FFDecoder: TFFDecoder;
    procedure FormCreate(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure btnStartClick(Sender: TObject);
    procedure btnStopClick(Sender: TObject);
    procedure btnPauseClick(Sender: TObject);
    procedure btnResumeClick(Sender: TObject);
    procedure btnWebSiteClick(Sender: TObject);
    procedure cboLogLevelChange(Sender: TObject);
    procedure chkPreviewClick(Sender: TObject);
    procedure btnInputFileClick(Sender: TObject);
    procedure btnFileInfoClick(Sender: TObject);
    procedure btnOutputFileClick(Sender: TObject);
    procedure FFEncoderProgress(Sender: TObject; AProgressInfo: PProgressInfo);
    procedure FFEncoderTerminate(Sender: TObject; const ATerminateInfo: TTerminateInfo);
    procedure FFLoggerLog(Sender: TObject; AThreadID: Integer;
      ALogLevel: TLogLevel; const ALogMsg: string);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  frmFFmpegCommand: TfrmFFmpegCommand;

implementation

{$R *.dfm}

uses
  IniFiles,
  ShellAPI,
  MyUtils;

const
  CLibAVPath = 'LibAV';

  SAppTitle = 'Demo of FFEncoder %s';
  SCaption = 'Demo of FFEncoder %s (FFmpegCommand) - Delphi FFmpeg VCL Components';
  SWebSiteC = 'http://www.CCAVC.com';
  SWebSiteE = 'http://www.DelphiFFmpeg.com';


  CHistoryCount = 25;

var
  SWebSite: string = SWebSiteE;

procedure SaveComboBoxValue(AComboBox: TComboBox);
var
  LIniFile: TMemIniFile;
  SL: TStringList;
  I: Integer;
  C: Integer;
begin
  SL := TStringList.Create;
  LIniFile := TMemIniFile.Create(ChangeFileExt(Application.ExeName, '.ini'){$IFDEF UNICODE}, TEncoding.UTF8{$ENDIF});
  try
    LIniFile.EraseSection(AComboBox.Name);
    C := 0;
    if Trim(AComboBox.Text) = '' then
      LIniFile.WriteString(AComboBox.Name, 'Item', '');
    for I := 0 to AComboBox.Items.Count - 1 do
    begin
      if (Trim(AComboBox.Items.Strings[I]) <> '') and
        (SL.IndexOf(Trim(AComboBox.Items.Strings[I])) < 0) then
      begin
        SL.Add(Trim(AComboBox.Items.Strings[I]));
        LIniFile.WriteString(AComboBox.Name, 'Item' + IntToStr(I), Trim(AComboBox.Items.Strings[I]));
        Inc(C);
      end;
      if C = CHistoryCount then
        Break;
    end;
    LIniFile.UpdateFile;
  finally
    SL.Free;
    LIniFile.Free;
  end;
end;

procedure LoadComboBoxValue(AComboBox: TComboBox);
var
  LIniFile: TMemIniFile;
  SL: TStringList;
  I: Integer;
begin
  SL := TStringList.Create;
  LIniFile := TMemIniFile.Create(ChangeFileExt(Application.ExeName, '.ini'){$IFDEF UNICODE}, TEncoding.UTF8{$ENDIF});
  try
    if LIniFile.SectionExists(AComboBox.Name) then
    begin
      LIniFile.ReadSectionValues(AComboBox.Name, SL);
      for I := 0 to SL.Count - 1 do
{$IFDEF VER140} // Delphi 6
        if (Trim(Copy(SL.Strings[I], Length(SL.Names[I]) + 2, MaxInt)) <> '') or (I = 0) then
          AComboBox.Items.Add(Trim(Copy(SL.Strings[I], Length(SL.Names[I]) + 2, MaxInt)));
{$ELSE}
        if (Trim(SL.ValueFromIndex[I]) <> '') or (I = 0) then
          AComboBox.Items.Add(Trim(SL.ValueFromIndex[I]));
{$ENDIF}
    end;
    if AComboBox.Items.Count > 0 then
      AComboBox.Text := AComboBox.Items.Strings[0];
  finally
    SL.Free;
    LIniFile.Free;
  end;
end;

procedure UpdateComboBox(AComboBox: TComboBox);
var
  I: Integer;
  S: string;
begin
  if Trim(AComboBox.Text) = '' then
    Exit;
  S := Trim(AComboBox.Text);
  for I := 0 to AComboBox.Items.Count - 1 do
    if SameText(Trim(AComboBox.Items.Strings[I]), S) then
    begin
      if I = 0 then
        Exit;
      AComboBox.Items.Delete(I);
      Break;
    end;
  AComboBox.Items.Insert(0, S);
  AComboBox.Text := S;
end;

procedure TfrmFFmpegCommand.FormCreate(Sender: TObject);
begin
  Application.Title := Format(SAppTitle, [FFEncoder.Version]);
  Self.Caption := Format(SCaption, [FFEncoder.Version]);

  if SysUtils.SysLocale.PriLangID = LANG_CHINESE then
    SWebSite := SWebSiteC
  else
    SWebSite := SWebSiteE;

  mmoLog.Text := SWebSite + #13#10#13#10 +
    'Compatible with: ffmpeg.exe [Input options] -i {Input file/URL} [Output options] {Output file/URL}' + #13#10#13#10;
  btnWebsite.Hint := SWebSite;
  btnWebsite.ShowHint := True;

  LoadComboBoxValue(cboInputOptions);
  LoadComboBoxValue(cboInputFile);
  LoadComboBoxValue(cboOutputOptions);
  LoadComboBoxValue(cboOutputFile);
  if (cboInputOptions.Text = '') and (cboInputFile.Text = '') and
     (cboOutputOptions.Text = '') and (cboOutputFile.Text = '') and
     (cboInputOptions.Items.Count <= 1) then
  begin
    cboInputOptions.Items.Add('-help');
    cboInputOptions.Items.Add('-version');
    cboInputOptions.Items.Add('-formats');
    cboInputOptions.Items.Add('-muxers');
    cboInputOptions.Items.Add('-demuxers');
    cboInputOptions.Items.Add('-devices');
    cboInputOptions.Items.Add('-codecs');
    cboInputOptions.Items.Add('-decoders');
    cboInputOptions.Items.Add('-encoders');
    cboInputOptions.Items.Add('-bsfs');
    cboInputOptions.Items.Add('-protocols');
    cboInputOptions.Items.Add('-filters');
    cboInputOptions.Items.Add('-pix_fmts');
    cboInputOptions.Items.Add('-layouts');
    cboInputOptions.Items.Add('-sample_fmts');
    cboInputOptions.Items.Add('-dispositions');
    cboInputOptions.Items.Add('-colors');
    cboInputOptions.Items.Add('-sources');
    cboInputOptions.Items.Add('-sinks');
    cboInputOptions.Items.Add('-hwaccels');
    cboInputOptions.Text := '-help';
  end;

  // disable all fpu exceptions(floating point exceptions):
  //   invalid operation, denormalized, divide by zero, overflow, underflow, inexact/precision
  FFEncoder.DisableFPUExceptions;
end;

procedure TfrmFFmpegCommand.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  with FFEncoder do
  begin
    // Clear the event handlers
    OnProgress := nil;
    OnTerminate := nil;

    // Break converting
    Stop;
  end;
  FFLogger.OnLog := nil;

  SaveComboBoxValue(cboInputOptions);
  SaveComboBoxValue(cboInputFile);
  SaveComboBoxValue(cboOutputOptions);
  SaveComboBoxValue(cboOutputFile);
end;

procedure TfrmFFmpegCommand.btnInputFileClick(Sender: TObject);
begin
  OpenDialog1.FileName := Trim(cboInputFile.Text);
  if (OpenDialog1.FileName <> '') and DirectoryExists(ExtractFilePath(OpenDialog1.FileName)) then
    OpenDialog1.InitialDir := ExtractFilePath(OpenDialog1.FileName);
  if OpenDialog1.Execute then
    cboInputFile.Text := OpenDialog1.FileName;
end;

procedure TfrmFFmpegCommand.btnFileInfoClick(Sender: TObject);
var
  S: string;
begin
  S := Trim(cboInputFile.Text);
  if S <> '' then
  begin
    if FileExists(S) then
    begin
      // Load dynamic link libraries
      if not FFDecoder.AVLibLoaded then
      begin
        // TPathFileName = type WideString;
        // FFDecoder.LoadAVLib(const APath: TPathFileName): Boolean;
        // APath: Full path indicates location of FFmpeg DLLs.
        //        It can be empty, let Windows search DLLs in current dir or environment <PATH>
        //if not FFDecoder.LoadAVLib(ExtractFilePath(Application.ExeName) + CLibAVPath) then
        // the routine ExePath() is implemented in unit MyUtils which returns WideString type
        // of ExtractFilePath(Application.ExeName)
        if not FFDecoder.LoadAVLib(ExePath + CLibAVPath) then
        begin
          mmoLog.Lines.Add(FFDecoder.LastErrMsg);
          Exit;
        end;
      end;
      if FFDecoder.LoadFile(S) then
      begin
        mmoLog.Lines.Add('');
        mmoLog.Lines.Add(FFDecoder.FileInfoText);
      end
      else
        mmoLog.Lines.Add(FFDecoder.LastErrMsg);
    end
    else
      mmoLog.Lines.Add(Format('File %s not exists', [S]));
  end;
end;

procedure TfrmFFmpegCommand.btnOutputFileClick(Sender: TObject);
begin
  SaveDialog1.FileName := Trim(cboOutputFile.Text);
  if (SaveDialog1.FileName <> '') and DirectoryExists(ExtractFilePath(SaveDialog1.FileName)) then
    SaveDialog1.InitialDir := ExtractFilePath(SaveDialog1.FileName);
  if SaveDialog1.Execute then
    cboOutputFile.Text := SaveDialog1.FileName;
end;

procedure TfrmFFmpegCommand.btnStartClick(Sender: TObject);
var
  LIndex: Integer;
  LInFileName: string;
  LOutFileName: string;
  LOptions: string;
begin
  UpdateComboBox(cboInputOptions);
  UpdateComboBox(cboInputFile);
  UpdateComboBox(cboOutputOptions);
  UpdateComboBox(cboOutputFile);

  // Load dynamic link libraries
  if not FFEncoder.AVLibLoaded then
  begin
    // TPathFileName = type WideString;
    // FFEncoder.LoadAVLib(const APath: TPathFileName): Boolean;
    // APath: Full path indicates location of FFmpeg DLLs.
    //        It can be empty, let Windows search DLLs in current dir or environment <PATH>
    //if not FFEncoder.LoadAVLib(ExtractFilePath(Application.ExeName) + CLibAVPath) then
    // the routine ExePath() is implemented in unit MyUtils which returns WideString type
    // of ExtractFilePath(Application.ExeName)
    if not FFEncoder.LoadAVLib(ExePath + CLibAVPath) then
    begin
      mmoLog.Lines.Add(FFEncoder.LastErrMsg);
      Exit;
    end;
  end;

  LInFileName := Trim(cboInputFile.Text);
  LOutFileName := Trim(cboOutputFile.Text);

  if (LOutFileName <> '') and FileExists(LOutFileName) then
  begin
    if Application.MessageBox(PChar(Format('The target file "%s" already exists, overwrite?', [LOutFileName])),
       'Confirm', MB_OKCANCEL + MB_ICONQUESTION) <> ID_OK then
    begin
      cboOutputFile.SetFocus;
      Exit;
    end;
  end;

  // ensure reset FFEncoder
  FFEncoder.ClearTasks;
  ProgressBar1.Position := 0;

  mmoLog.Lines.Add('');

  LOptions := Trim(cboInputOptions.Text);
  if LInFileName <> '' then
    LOptions := LOptions + Format(' -i "%s"', [LInFileName]);
  if Trim(cboOutputOptions.Text) <> '' then
    LOptions := LOptions + ' ' + Trim(cboOutputOptions.Text);
  if LOutFileName <> '' then
    LOptions := LOptions + Format(' -y "%s"', [LOutFileName]); // -y overwrite output files

  // add a task
  LIndex := FFEncoder.AddTask(LOptions);
  if LIndex < 0 then
  begin
    mmoLog.Lines.Add('');
    mmoLog.Lines.Add('***Add task error: ' + FFEncoder.LastErrMsg);
    mmoLog.Lines.Add('');
    Exit;
  end;

  mmoLog.Lines.Add('');
  mmoLog.Lines.Add('***Ready to go!');
  mmoLog.Lines.Add('');

  // set status of buttons
  btnStart.Enabled := False;
  btnStop.Enabled := True;
  btnPause.Enabled := True;
  btnResume.Enabled := False;
  btnStop.SetFocus;

  FFEncoder.PreviewVideo := chkPreview.Checked;
  FFEncoder.PreviewAudio := chkPreview.Checked;

  // procedure Start(ATaskCount: Integer);
  // ATaskCount: >  0, perform tasks in asynchronous/non-blocking mode
  //             >  1, perform multiple tasks in the same time
  //             <= 0, perform tasks in synchronous/blocking mode
  FFEncoder.Start(1);
end;

procedure TfrmFFmpegCommand.btnStopClick(Sender: TObject);
begin
  btnStop.Enabled := False;
  FFEncoder.Stop; // only works in asynchronous mode
end;

procedure TfrmFFmpegCommand.btnPauseClick(Sender: TObject);
begin
  btnPause.Enabled := False;
  btnResume.Enabled := True;
  FFEncoder.Pause; // only works in asynchronous mode
end;

procedure TfrmFFmpegCommand.btnResumeClick(Sender: TObject);
begin
  btnPause.Enabled := True;
  btnResume.Enabled := False;
  FFEncoder.Resume; // only works in asynchronous mode
end;

procedure TfrmFFmpegCommand.btnWebSiteClick(Sender: TObject);
  function FromEXE: string;
  var
    S: string;
  begin
    S := ChangeFileExt(ExtractFileName(Application.ExeName), '');
    S := StringReplace(S, '[', '', [rfReplaceAll]);
    S := StringReplace(S, ']', '', [rfReplaceAll]);
    S := StringReplace(S, ' ', '_', [rfReplaceAll]);
    Result := '/?from=exe_' + S;
  end;
begin
  ShellExecute(Application.Handle, 'Open',   {Do not Localize}
    PChar(LowerCase(SWebSite + FromEXE)), '',
    PChar(ExtractFilePath(Application.ExeName)), 1);
end;

procedure TfrmFFmpegCommand.cboLogLevelChange(Sender: TObject);
begin
  FFLogger.LogLevel := TLogLevel(cboLogLevel.ItemIndex);
end;

procedure TfrmFFmpegCommand.chkPreviewClick(Sender: TObject);
begin
  FFEncoder.PreviewVideo := chkPreview.Checked;
  FFEncoder.PreviewAudio := chkPreview.Checked;
end;

procedure TfrmFFmpegCommand.FFEncoderProgress(Sender: TObject;
  AProgressInfo: PProgressInfo);
begin
  // OnProgress event handler
{
  PProgressInfo = ^TProgressInfo;
  TProgressInfo = record
    TaskIndex: Integer;     // index of converting tasks
    FileIndex: Integer;     // index of input files in the current task
    FrameNumber: Integer;   // current frame number
    FPS: Integer;           // video converting speed, frames per second, not valid when only audio output
    Quality: Single;        // quality
    BitRate: Single;        // bitrate
    CurrentSize: Int64;     // current output file size in bytes
    CurrentDuration: Int64; // current duration time in microsecond
    TotalDuration: Int64;   // total output duration time in microsecond
  end;
}
  with AProgressInfo^ do
  begin
    lblStatus.Caption := Format('Frame number: %d; FPS: %d; Size: %d; Time: %d',
        [FrameNumber, FPS, CurrentSize, CurrentDuration]);
    if TotalDuration > 0 then
      ProgressBar1.Position := CurrentDuration * 100 div TotalDuration;
  end;
end;

procedure TfrmFFmpegCommand.FFEncoderTerminate(Sender: TObject;
  const ATerminateInfo: TTerminateInfo);
begin
  // OnTerminate event handler
{
  TTerminateInfo = record
    TaskIndex: Integer;     // index of converting tasks, (-1) means all tasks are terminated
    Finished: Boolean;      // True means converted success, False means converting broken
    Exception: Boolean;     // True means Exception occured, False please ignore
    ExceptionMsg: string;   // Exception message
  end;
}
  if ATerminateInfo.TaskIndex < 0 then
  begin
    // set status of buttons
    btnStart.Enabled := True;
    btnStop.Enabled := False;
    btnPause.Enabled := False;
    btnResume.Enabled := False;
  end
  else if ATerminateInfo.Finished then
  begin // TaskIndex task converted success
    ProgressBar1.Position := 100;
  end;
  if ATerminateInfo.Exception then // Exception occured, show exception message
    Application.MessageBox(PChar(ATerminateInfo.ExceptionMsg), PChar(Application.Title), MB_OK + MB_ICONWARNING);
end;

procedure TfrmFFmpegCommand.FFLoggerLog(Sender: TObject; AThreadID: Integer;
  ALogLevel: TLogLevel; const ALogMsg: string);
begin
  mmoLog.Lines.Add(ALogMsg);
end;

end.
