unit DecoderFrm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ComCtrls, ExtCtrls, ExtDlgs, FFLoad, FFLog,
  FFBaseComponent, FFDecode;

type
  TfrmDecoder = class(TForm)
    lblPTS: TLabel;
    Image1: TImage;
    btnOpen: TButton;
    btnSave: TButton;
    btnNextFrame: TButton;
    btnPreviousFrame: TButton;
    btnNextKF: TButton;
    btnPreviousKF: TButton;
    btnWebSite: TButton;
    btnExit: TButton;
    cboLogLevel: TComboBox;
    mmoLog: TMemo;
    TrackBar1: TTrackBar;
    FFDecoder: TFFDecoder;
    FFLogger: TFFLogger;
    OpenDialog1: TOpenDialog;
    SavePictureDialog1: TSavePictureDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure Image1Click(Sender: TObject);
    procedure TrackBar1Change(Sender: TObject);
    procedure btnOpenClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
    procedure btnNextFrameClick(Sender: TObject);
    procedure btnPreviousFrameClick(Sender: TObject);
    procedure btnNextKFClick(Sender: TObject);
    procedure btnPreviousKFClick(Sender: TObject);
    procedure btnWebSiteClick(Sender: TObject);
    procedure btnExitClick(Sender: TObject);
    procedure cboLogLevelChange(Sender: TObject);
    procedure FFLoggerLog(Sender: TObject; AThreadID: Integer;
      ALogLevel: TLogLevel; const ALogMsg: string);
  private
    { Private declarations }
    FBitmap: TBitmap;
    FDuration: Int64;
    FFrames: Integer;
    FChanging: Boolean;
    FCurrentPTS: Int64;
    FLastPosition: Integer;
    procedure DoActualFormClick(Sender: TObject);
    procedure ClearFrameView;
    procedure SetupTrackBar;
    procedure DrawCurrentFrame;
    procedure ReadAndDrawFrame(APTS: Int64);
    procedure ReadAndDrawNextFrame;
    procedure ReadAndDrawPreviousFrame;
  public
    { Public declarations }
  end;

var
  frmDecoder: TfrmDecoder;

implementation

{$R *.dfm}

uses
  Jpeg,
  ShellAPI,
  MyUtils;

const
  CLibAVPath = 'LibAV';

  SAppTitle = 'Demo of FFDecoder %s';
  SCaption = 'Demo of FFDecoder %s - Delphi FFmpeg VCL Components';
  SWebSiteC = 'http://www.CCAVC.com';
  SWebSiteE = 'http://www.DelphiFFmpeg.com';
  SViewActual = 'Click to view current picture in actual size';


  CDialogOptions = [ofHideReadOnly, ofFileMustExist, ofEnableSizing];
  CPictureFiles = '*.BMP;*.GIF;*.JPEG;*.JPG;*.PNG;';
  CAudioFiles = '*.AAC;*.AC3;*.APE;*.DTS;*.FLAC;*.M4A;*.MKA;*.MP2;*.MP3;' +
      '*.MPA;*.MPC;*.OFR;*.OGG;*.RA;*.TTA;*.WAV;*.WMA;';
  CVideoFiles = '*.3GP;*.ASF;*.AVI;*.AVM;*.AVS;*.DAT;*.FLV;*.MKV;*.MOV;' +
      '*.MP4;*.MPEG;*.MPG;*.NSV;*.OGM;*.RM;*.RMVB;*.TP;*.TS;*.VOB;*.WMV;';
  CDialogFilter =
      'Video/Audio/Picture Files|' + CVideoFiles + CAudioFiles + CPictureFiles +
      '|Video Files|' + CVideoFiles +
      '|Audio Files|' + CAudioFiles +
      '|Picture Files|' + CPictureFiles +
      '|All Files|*.*';

var
  SWebSite: string = SWebSiteE;

{ Utils }

function DurationToStr(ADuration: Int64): string;
begin
  Result := Format('%.2d:%.2d:%.2d.%.3d',
    [ADuration div AV_TIME_BASE div 60 div 60,
     ADuration div AV_TIME_BASE div 60 mod 60,
     ADuration div AV_TIME_BASE mod 60,
     ADuration mod AV_TIME_BASE * 1000 div AV_TIME_BASE]);
end;

// scale source rect into destination rect
function FitRect(ASrcRect, ADestRect: TRect): TRect;
var
  LWFactor: Double;
  LHFactor: Double;
  LSpace: Integer;
begin
  LWFactor := (ASrcRect.Right - ASrcRect.Left) / (ADestRect.Right - ADestRect.Left);
  LHFactor := (ASrcRect.Bottom - ASrcRect.Top) / (ADestRect.Bottom - ADestRect.Top);
  if LWFactor < LHFactor then
  begin
    Result.Top := ADestRect.Top;
    Result.Bottom := ADestRect.Bottom;
    LSpace := Round(((ADestRect.Right - ADestRect.Left) - (ASrcRect.Right - ASrcRect.Left) / LHFactor) / 2);
    Result.Left := ADestRect.Left + LSpace;
    Result.Right := ADestRect.Right - LSpace;
  end
  else
  begin
    Result.Left := ADestRect.Left;
    Result.Right := ADestRect.Right;
    LSpace := Round(((ADestRect.Bottom - ADestRect.Top) - (ASrcRect.Bottom - ASrcRect.Top) / LWFactor) / 2);
    Result.Top := ADestRect.Top + LSpace;
    Result.Bottom := ADestRect.Bottom - LSpace;
  end;
end;

// stretch draw bitmap on image with fit scale
procedure FitDraw(ACanvas: TCanvas; ABitmap: TBitmap);
var
  R: TRect;
begin
  if ABitmap.Width * ABitmap.Height = 0 then Exit;
  R := ACanvas.ClipRect;
  // calculate scaled rect then draw bitmap on image
{$IFDEF DRAW_FRAME}
  InflateRect(R, -1, -1);
{$ENDIF}
  R := FitRect(ABitmap.Canvas.ClipRect, R);
  ACanvas.StretchDraw(R, ABitmap);
{$IFDEF DRAW_FRAME}
  ACanvas.Brush.Color := clGreen;
  InflateRect(R, 1, 1);
  ACanvas.FrameRect(R);
{$ENDIF}
end;

procedure TfrmDecoder.FormCreate(Sender: TObject);
begin
  Application.Title := Format(SAppTitle, [FFDecoder.Version]);
  Self.Caption := Format(SCaption, [FFDecoder.Version]);
  if SysUtils.SysLocale.PriLangID = LANG_CHINESE then
    SWebSite := SWebSiteC
  else
    SWebSite := SWebSiteE;

  mmoLog.Text := SWebSite + #13#10#13#10;
  btnWebsite.Hint := SWebSite;
  btnWebsite.ShowHint := True;
  Image1.Hint := SViewActual;
  Image1.ShowHint := True;

  // open dialog setting
  OpenDialog1.Options := CDialogOptions;
  OpenDialog1.Filter := CDialogFilter;

  // save dialog setting
  SavePictureDialog1.Options := [ofOverwritePrompt, ofHideReadOnly, ofExtensionDifferent, ofPathMustExist, ofEnableSizing];
  SavePictureDialog1.Filter := 'Bitmaps (*.bmp)|*.bmp|JPEG Image File |*.jpg;*.jpeg';
  SavePictureDialog1.DefaultExt := 'bmp';

  // create instance
  FBitmap := TBitmap.Create;

  // clear frame view
  ClearFrameView;

end;

procedure TfrmDecoder.FormResize(Sender: TObject);
begin
  // setup track bar
  SetupTrackBar;
  // adjust Image
  Image1.Picture.Graphic.Width := Image1.Width;
  Image1.Picture.Graphic.Height := Image1.Height;
  Image1.Canvas.Brush.Color := clSkyBlue;
  Image1.Canvas.FillRect(Image1.Canvas.ClipRect);
  // redraw the Bitmap
  if FBitmap.Width <> 0 then
    FitDraw(Image1.Canvas, FBitmap)
  else
    ClearFrameView;
end;

procedure TfrmDecoder.FormDestroy(Sender: TObject);
begin
  FBitmap.Free;
end;

procedure TfrmDecoder.ClearFrameView;
begin
  // reset global variables
  FDuration := 0;
  FFrames := 0;
  FCurrentPTS := 0;
  // clear cached frame picture
  FBitmap.Width := 0;
  FBitmap.Height := 0;
  // fill background
  Image1.Canvas.Brush.Color := clSkyBlue;
  Image1.Canvas.FillRect(Image1.Canvas.ClipRect);
  // diable track bar
  TrackBar1.Enabled := False;
  TrackBar1.Position := 0;
  SetupTrackBar;
  // update pts caption
  lblPTS.Caption := DurationToStr(0);
end;

procedure TfrmDecoder.SetupTrackBar;
var
  LEnabled: Boolean;
  LPosRate: Double;
begin
  LEnabled := TrackBar1.Enabled;

  TrackBar1.Enabled := False;

  if TrackBar1.Max > 0 then
    LPosRate := TrackBar1.Position / TrackBar1.Max
  else
    LPosRate := 0;

  if FFrames > 0 then
{$IF CompilerVersion >= 20} // Delphi 2009 or above
    TrackBar1.Max := FFrames;
{$ELSE}
    if FFrames > $7FFF then
      TrackBar1.Max := $7FFF
    else
      TrackBar1.Max := FFrames;
{$IFEND}

  if TrackBar1.Max > 10 then
    TrackBar1.Frequency := TrackBar1.Max div 10
  else
    TrackBar1.Frequency := 1;

  if TrackBar1.Max < 10 then
    TrackBar1.PageSize := 2
  else if TrackBar1.Max < 100 then
    TrackBar1.PageSize := 5
  else
    TrackBar1.PageSize := TrackBar1.Max div 20;

  TrackBar1.Position := Round(TrackBar1.Max * LPosRate);
  FLastPosition := TrackBar1.Position;

  TrackBar1.Enabled := LEnabled;
  TrackBar1.SliderVisible := LEnabled;
end;

procedure TfrmDecoder.DrawCurrentFrame;
var
  LPosition: Integer;
begin
  // update current pts and current position
  if FFDecoder.FrameInfo.PTS <> AV_NOPTS_VALUE then
    FCurrentPTS := FFDecoder.FrameInfo.PTS
  else
    FCurrentPTS := 0;
  lblPTS.Caption := DurationToStr(FCurrentPTS);
  if not FChanging and (FDuration > 0) then
  begin
    FChanging := True;
    try
      LPosition := Round(TrackBar1.Max * FCurrentPTS / FDuration);
      if LPosition > TrackBar1.Max then
        LPosition := TrackBar1.Max;
      TrackBar1.Position := LPosition;
    finally
      FChanging := False;
    end;
  end;
  // copy the frame to bitmap
  if FFDecoder.CopyToBitmap(FBitmap) then
    FitDraw(Image1.Canvas, FBitmap)
  else
    mmoLog.Lines.Add('***CopyToBitmap failed: ' + FFDecoder.LastErrMsg);
end;

procedure TfrmDecoder.ReadAndDrawFrame(APTS: Int64);
begin
  // try to seek the frame
  if FFDecoder.Seek(APTS) or
    FFDecoder.Seek(APTS, [sfBackward]) or
    FFDecoder.Seek(APTS, [sfAny]) then
  begin
    // decode the next frame
    while FFDecoder.Decode do
    begin
      // if the frame is the one we desire
      if (FFDecoder.FrameInfo.PTS >= APTS) or (APTS <= 0) then
      begin
        DrawCurrentFrame;
        Exit;
      end;
    end;
    mmoLog.Lines.Add('***Decode failed: ' + FFDecoder.LastErrMsg);
  end
  else
    mmoLog.Lines.Add('***SeekFrame failed: ' + FFDecoder.LastErrMsg);
end;

procedure TfrmDecoder.ReadAndDrawNextFrame;
begin
  if FFDecoder.Decode then
    DrawCurrentFrame
  else
    mmoLog.Lines.Add('***Decode failed: ' + FFDecoder.LastErrMsg);
end;

procedure TfrmDecoder.ReadAndDrawPreviousFrame;
begin
  if FFDecoder.DecodePreviousFrame then
    DrawCurrentFrame
  else
    mmoLog.Lines.Add('***Decode failed: ' + FFDecoder.LastErrMsg);
end;

procedure TfrmDecoder.DoActualFormClick(Sender: TObject);
begin
  if Assigned(Sender) and (Sender is TControl) and ((Sender as TControl).Parent is TForm) then
    ((Sender as TControl).Parent as TForm).Close;
end;

procedure TfrmDecoder.Image1Click(Sender: TObject);
var
  F: TForm;
  I: TImage;
  B: TButton;
begin
  // sanity check to ensure the current frame is available
  if (FBitmap.Width = 0) or (FBitmap.Height = 0) then
    Exit;

  // show current frame picture in actual size
  F := TForm.Create(Self);
  try
    // create a Button to response Enter and ESC key-press
    B := TButton.Create(F);
    with B do
    begin
      Parent := F;
      Default := True;
      Cancel := True;
      ModalResult := mrCancel;
      Top := 0;
      Left := 0;
      Width := 0;
      Height := 0;
      Visible := True;
    end;
    // create an Image to display the frame
    I := TImage.Create(F);
    with I do
    begin
      Parent := F;
      OnClick := Self.DoActualFormClick;
      Visible := True;
      Left := 0;
      Top := 0;
      Width := FBitmap.Width;
      Height := FBitmap.Height;
      Picture.Assign(FBitmap);
    end;
    // setup the form and then show it
    with F do
    begin
      BorderIcons := [biSystemMenu];
      BorderStyle := bsSingle;
      Position := poScreenCenter;
      Caption := Format('%dx%d @ %s', [FBitmap.Width, FBitmap.Height,
        DurationToStr(FCurrentPTS)]);;
      ClientWidth := I.Width;
      ClientHeight := I.Height;
      ShowModal;
    end;
  finally
    F.Free;
  end;
end;

procedure TfrmDecoder.TrackBar1Change(Sender: TObject);
var
  LDesirePTS: Int64;
begin
  // check the situation for seeking
  if FChanging or not TrackBar1.Enabled or (FFDecoder.VideoStreamCount <= 0) or
    (FLastPosition = TrackBar1.Position) then
    Exit;

  FChanging := True;
  try
    // position rate
    if TrackBar1.Max > 0 then
      LDesirePTS := Round(TrackBar1.Position / TrackBar1.Max * FDuration)
    else
      LDesirePTS := 0;

    if TrackBar1.Position = FLastPosition + 1 then
      // next frame
      ReadAndDrawNextFrame
    else if TrackBar1.Position = FLastPosition - 1 then
      // previous frame
      ReadAndDrawPreviousFrame
    else
      // special frame
      ReadAndDrawFrame(LDesirePTS);
  finally
    FLastPosition := TrackBar1.Position;
    FChanging := False;
  end;
end;

procedure TfrmDecoder.btnOpenClick(Sender: TObject);
var
  I: Integer;
begin
  // Load dynamic link libraries
  if not FFDecoder.AVLibLoaded then
  begin
    // TPathFileName = type WideString;
    // FFDecoder.LoadAVLib(const APath: TPathFileName): Boolean;
    // APath: Full path indicates location of FFmpeg DLLs.
    //        It can be empty, let Windows search DLLs in current dir or environment <PATH>
    //if not FFDecoder.LoadAVLib(ExtractFilePath(Application.ExeName) + CLibAVPath) then
    // the routine ExePath() is implemented in unit MyUtils which returns WideString type
    // of ExtractFilePath(Application.ExeName)
    if not FFDecoder.LoadAVLib(ExePath + CLibAVPath) then
    begin
      mmoLog.Lines.Add(FFDecoder.LastErrMsg);
      Exit;
    end;
  end;

  if not OpenDialog1.Execute then
    // cancel open file
    Exit;

  // clear frame view and reset global variables
  ClearFrameView;

  if not FFDecoder.LoadFile(OpenDialog1.FileName) then
  begin // load av file failed
    mmoLog.Lines.Add('');
    mmoLog.Lines.Add('***File load error: ' + FFDecoder.LastErrMsg);
    mmoLog.Lines.Add('');
    Exit;
  end;

  // show FileInfoText
  mmoLog.Lines.Add('***FFDecoder.FileInfoText');
  mmoLog.Lines.Add(Trim(FFDecoder.FileInfoText));
  mmoLog.Lines.Add('');

  // show file size
  mmoLog.Lines.Add(Format('File Size: %d',[FFDecoder.FileSize]));

  // show file duration
  with FFDecoder.FileStreamInfo do
  begin
    if Duration <> AV_NOPTS_VALUE then
      mmoLog.Lines.Add(Format('File Duration: %s', [DurationToStr(Duration)]))
    else
      mmoLog.Lines.Add('File Duration is not preseted');
  end;

  // show stream information
  for I := 0 to FFDecoder.StreamCount - 1 do
  begin
    if FFDecoder.IsVideoStream(I) then
    begin
      // video stream
      mmoLog.Lines.Add(Format('Stream #%d: Video', [I]));
      with FFDecoder.VideoStreamInfos[I] do
      begin
        // video stream duration
        if Duration <> AV_NOPTS_VALUE then
          mmoLog.Lines.Add(Format('  Video Length: %s', [DurationToStr(DurationScaled)]))
        else
          mmoLog.Lines.Add('  Video Length is not preseted');
        // video frame rate
        mmoLog.Lines.Add(Format('  Video Frame Rate: %f', [FrameRate.num / FrameRate.den]));
        // video bit rate
        if BitRate > 0 then
          mmoLog.Lines.Add(Format('  Video Bit Rate: %d kb', [BitRate div 1000]))
        else
          mmoLog.Lines.Add('  Video Bit Rate is not preseted');
        // video frame size
        mmoLog.Lines.Add(Format('  Frame size (width X height): %dx%d', [Width, Height]));
      end;
    end
    else if FFDecoder.IsAudioStream(I) then
    begin
      // audio stream
      mmoLog.Lines.Add(Format('Stream #%d: Audio', [I]));
      with FFDecoder.AudioStreamInfos[I] do
      begin
        // audio stream duration
        if Duration <> AV_NOPTS_VALUE then
          mmoLog.Lines.Add(Format('  Audio Length: %s', [DurationToStr(DurationScaled)]))
        else
          mmoLog.Lines.Add('  Audio Length is not preseted');
        // audio sample rate
        mmoLog.Lines.Add(Format('  Audio Sample Rate: %d hz', [SampleRate]));
        // audio bit rate
        if BitRate > 0 then
          mmoLog.Lines.Add(Format('  Audio Bit Rate: %d kb', [BitRate div 1000]))
        else
          mmoLog.Lines.Add('  Audio Bit Rate is not preseted');
      end;
    end;
  end;

  with FFDecoder do
  begin
    // whether we can seek or not
    TrackBar1.Enabled := (FileStreamInfo.Duration <> AV_NOPTS_VALUE) and (FormatName <> 'image2') and
                         (VideoStreamCount > 0);

    // file duration
    if FileStreamInfo.Duration <> AV_NOPTS_VALUE then
      FDuration := FileStreamInfo.Duration;

    // calculate Frames of the first video stream, use it for the track bar max position
    if (VideoStreamCount > 0) and (FirstVideoStreamInfo.FrameRate.den > 0) then
    begin
      if FirstVideoStreamInfo.DurationScaled > 0 then
        FFrames := Round(FirstVideoStreamInfo.DurationScaled / AV_TIME_BASE *
          FirstVideoStreamInfo.FrameRate.num / FirstVideoStreamInfo.FrameRate.den)
      else if FileStreamInfo.Duration <> AV_NOPTS_VALUE then
        FFrames := Round(FileStreamInfo.Duration / AV_TIME_BASE *
          FirstVideoStreamInfo.FrameRate.num / FirstVideoStreamInfo.FrameRate.den);
    end;
  end;

  // if video stream available, then read and draw the first frame
  if FFDecoder.VideoStreamCount > 0 then
    ReadAndDrawNextFrame;

  // setup track bar for seeking
  SetupTrackBar;
end;

procedure TfrmDecoder.btnSaveClick(Sender: TObject);
begin
  SavePictureDialog1.FileName := ChangeFileExt(ExtractFileName(OpenDialog1.FileName), '');
  if (FBitmap.Width <> 0) and SavePictureDialog1.Execute then
  begin
    if SameText(ExtractFileExt(SavePictureDialog1.FileName), '.bmp') then
      FBitmap.SaveToFile(SavePictureDialog1.FileName)
    else
      with TJPEGImage.Create do
      try
        Assign(FBitmap);
        SaveToFile(SavePictureDialog1.FileName)
      finally
        Free;
      end;
  end;
end;

procedure TfrmDecoder.btnNextFrameClick(Sender: TObject);
begin
  // same as ReadAndDrawNextFrame()
  if FFDecoder.Decode then
    DrawCurrentFrame
  else
    mmoLog.Lines.Add('***Decode failed: ' + FFDecoder.LastErrMsg);
end;

procedure TfrmDecoder.btnPreviousFrameClick(Sender: TObject);
begin
  // same as ReadAndDrawPreviousFrame()
  if FFDecoder.DecodePreviousFrame then
    DrawCurrentFrame
  else
    mmoLog.Lines.Add('***Decode failed: ' + FFDecoder.LastErrMsg);
end;

procedure TfrmDecoder.btnNextKFClick(Sender: TObject);
begin
  if FFDecoder.DecodeNextKeyFrame then
    DrawCurrentFrame
  else
    mmoLog.Lines.Add('***Decode failed: ' + FFDecoder.LastErrMsg);
end;

procedure TfrmDecoder.btnPreviousKFClick(Sender: TObject);
begin
  if FFDecoder.DecodePreviousKeyFrame then
    DrawCurrentFrame
  else
    mmoLog.Lines.Add('***Decode failed: ' + FFDecoder.LastErrMsg);
end;

procedure TfrmDecoder.btnWebSiteClick(Sender: TObject);
  function FromEXE: string;
  var
    S: string;
  begin
    S := ChangeFileExt(ExtractFileName(Application.ExeName), '');
    S := StringReplace(S, '[', '', [rfReplaceAll]);
    S := StringReplace(S, ']', '', [rfReplaceAll]);
    S := StringReplace(S, ' ', '_', [rfReplaceAll]);
    Result := '/?from=exe_' + S;
  end;
begin
  ShellExecute(Application.Handle, 'Open',   {Do not Localize}
    PChar(LowerCase(SWebSite + FromEXE)), '',
    PChar(ExtractFilePath(Application.ExeName)), 1);
end;

procedure TfrmDecoder.btnExitClick(Sender: TObject);
begin
  Close;
end;

procedure TfrmDecoder.cboLogLevelChange(Sender: TObject);
begin
  // TLogLevel = (llQuiet, llPanic, llFatal, llError, llWarning, llInfo, llVerbose, llDebug, llTrace);
  FFLogger.LogLevel := TLogLevel(cboLogLevel.ItemIndex);
end;

procedure TfrmDecoder.FFLoggerLog(Sender: TObject; AThreadID: Integer;
  ALogLevel: TLogLevel; const ALogMsg: string);
begin
  mmoLog.Lines.Add(ALogMsg);
end;

end.
