unit OptionFrm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ComCtrls, ExtCtrls, FFBaseComponent, FFEncode, FFDecode;

{$DEFINE QUICK_CUT}

type
  TfrmOption = class(TForm)
    lblOutPath: TLabel;
    grpAVFileInfo: TGroupBox;
    mmoAVFileInfo: TMemo;
    grpFormats: TRadioGroup;
    grpNorm: TRadioGroup;
    grpAudioChannels: TRadioGroup;
    grpAudioSampleRate: TRadioGroup;
    grpAudioVolume: TGroupBox;
    chkAudioVolume: TCheckBox;
    edtAudioVolume: TEdit;
    grpFrameSize: TGroupBox;
    cboFrameSize: TComboBox;
    grpFrameRate: TRadioGroup;
    grpAspectRatio: TRadioGroup;
    grpBitRate: TGroupBox;
    lblVideoBitRate: TLabel;
    edtVideoBitRate: TEdit;
    lblAudioBitRate: TLabel;
    edtAudioBitRate: TEdit;
    grpAudioCodec: TRadioGroup;
    grpVideoCodec: TRadioGroup;
    grpPadCrop: TGroupBox;
    lblTop: TLabel;
    lblBottom: TLabel;
    lblLeft: TLabel;
    lblRight: TLabel;
    chkCrop: TCheckBox;
    edtCropTop: TEdit;
    edtCropBottom: TEdit;
    edtCropLeft: TEdit;
    edtCropRight: TEdit;
    chkPad: TCheckBox;
    edtPadTop: TEdit;
    edtPadBottom: TEdit;
    edtPadLeft: TEdit;
    edtPadRight: TEdit;
    grpCutClip: TGroupBox;
    lblPTS: TLabel;
    Image1: TImage;
    TrackBar1: TTrackBar;
    btnStartTime: TButton;
    edtStartTime: TEdit;
    btnEndTime: TButton;
    edtEndTime: TEdit;
    btnNextFrame: TButton;
    chkCutClip: TCheckBox;
    grpHook: TGroupBox;
    chkVideoInputHook: TCheckBox;
    chkVideoOutputHook: TCheckBox;
    chkAudioHook: TCheckBox;
    grpVideoFilter: TGroupBox;
    chkVertFlip: TCheckBox;
    chkHoriFlip: TCheckBox;
    chkNegate: TCheckBox;
    chkWaterMark: TCheckBox;
    chkDeinterlace: TCheckBox;
    chkRotate: TCheckBox;
    edtRotate: TEdit;
    chkBrightness: TCheckBox;
    trbBrightness: TTrackBar;
    btnBrightness: TButton;
    chkSaturation: TCheckBox;
    trbSaturation: TTrackBar;
    btnSaturation: TButton;
    chkHue: TCheckBox;
    trbHue: TTrackBar;
    btnHue: TButton;
    chkOverwrite: TCheckBox;
    edtOutPath: TEdit;
    btnOutPath: TButton;
    btnOk: TButton;
    btnCancel: TButton;
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormDestroy(Sender: TObject);
    procedure chkAudioVolumeClick(Sender: TObject);
    procedure Image1Click(Sender: TObject);
    procedure TrackBar1Change(Sender: TObject);
    procedure btnStartTimeClick(Sender: TObject);
    procedure btnEndTimeClick(Sender: TObject);
    procedure btnNextFrameClick(Sender: TObject);
    procedure btnOutPathClick(Sender: TObject);
    procedure chkRotateClick(Sender: TObject);
    procedure chkCropClick(Sender: TObject);
    procedure chkPadClick(Sender: TObject);
    procedure chkBrightnessClick(Sender: TObject);
    procedure chkSaturationClick(Sender: TObject);
    procedure chkHueClick(Sender: TObject);
    procedure btnBrightnessClick(Sender: TObject);
    procedure btnSaturationClick(Sender: TObject);
    procedure btnHueClick(Sender: TObject);
  private
    FDecoder: TFFDecoder;
    FBitmap: TBitmap;
    FDuration: Int64;
    FFrames: Integer;
    FChanging: Boolean;
    FCurrentPTS: Int64;
    FLastPosition: Integer;
    FFirstPTS: Int64;
    FStartTime: Int64;
    FEndTime: Int64;
    FVideoStreamIndex: Integer;
    FAudioStreamIndex: Integer;
    procedure DoActualFormClick(Sender: TObject);
    procedure ClearFrameView(AResetCutClip: Boolean);
    procedure SetupTrackBar;
    procedure DrawCurrentFrame;
    procedure ReadAndDrawNextFrame;
    function GetClipStartTime: Int64;
    function GetClipTimeLength: Int64;
    procedure SetDecoder(const Value: TFFDecoder);
    function GetEventOptions: TEventOptions;
    function GetOptions: string;
  public
    property Decoder: TFFDecoder read FDecoder write SetDecoder;
    property EventOptions: TEventOptions read GetEventOptions;
    property Options: string read GetOptions;
  end;

//var
//  frmOption: TfrmOption;

implementation

{$R *.dfm}

uses
  ShlObj,
  ActiveX;

const
  CWaterMark = 'watermark.png';

  SSameAsInput = '<same as input>';
  SViewActual = 'Click to view current picture in actual size';

{$IF CompilerVersion <= 18.5} // Delphi 2007 and olders
type
  // Delphi 6 undefined
  // Delphi 7 to Delphi 2007 defined incorrectly, SizeOf(NativeInt) = 8, thus re-define it
  NativeInt = Integer;
  NativeUInt = Cardinal;
{$IFEND}

{ Utils }

function DurationToStr(ADuration: Int64): string;
begin
  Result := Format('%.2d:%.2d:%.2d.%.3d',
    [ADuration div AV_TIME_BASE div 60 div 60,
     ADuration div AV_TIME_BASE div 60 mod 60,
     ADuration div AV_TIME_BASE mod 60,
     ADuration mod AV_TIME_BASE * 1000 div AV_TIME_BASE]);
end;

// scale source rect into destination rect
function FitRect(ASrcRect, ADestRect: TRect): TRect;
var
  LWFactor: Double;
  LHFactor: Double;
  LSpace: Integer;
begin
  LWFactor := (ASrcRect.Right - ASrcRect.Left) / (ADestRect.Right - ADestRect.Left);
  LHFactor := (ASrcRect.Bottom - ASrcRect.Top) / (ADestRect.Bottom - ADestRect.Top);
  if LWFactor < LHFactor then
  begin
    Result.Top := ADestRect.Top;
    Result.Bottom := ADestRect.Bottom;
    LSpace := Round(((ADestRect.Right - ADestRect.Left) - (ASrcRect.Right - ASrcRect.Left) / LHFactor) / 2);
    Result.Left := ADestRect.Left + LSpace;
    Result.Right := ADestRect.Right - LSpace;
  end
  else
  begin
    Result.Left := ADestRect.Left;
    Result.Right := ADestRect.Right;
    LSpace := Round(((ADestRect.Bottom - ADestRect.Top) - (ASrcRect.Bottom - ASrcRect.Top) / LWFactor) / 2);
    Result.Top := ADestRect.Top + LSpace;
    Result.Bottom := ADestRect.Bottom - LSpace;
  end;
end;

// stretch draw bitmap on image with fit scale
procedure FitDraw(ACanvas: TCanvas; ABitmap: TBitmap);
var
  R: TRect;
begin
  if ABitmap.Width * ABitmap.Height = 0 then Exit;
  R := ACanvas.ClipRect;
  // calculate scaled rect then draw bitmap on image
{$IFDEF DRAW_FRAME}
  InflateRect(R, -1, -1);
{$ENDIF}
  R := FitRect(ABitmap.Canvas.ClipRect, R);
  ACanvas.StretchDraw(R, ABitmap);
{$IFDEF DRAW_FRAME}
  ACanvas.Brush.Color := clGreen;
  InflateRect(R, 1, 1);
  ACanvas.FrameRect(R);
{$ENDIF}
end;

function SelectDirCBEx(Wnd: HWND; uMsg: UINT; lParam, lpData: LPARAM): Integer stdcall;
var
  LRect: TRect;
  cx, cy, w, h, x, y: Integer;
begin
  if (uMsg = BFFM_INITIALIZED) then
  begin
    if (lpData <> 0) then
      SendMessage(Wnd, BFFM_SETSELECTION, Integer(True), lpData);
    cx := GetSystemMetrics(SM_CXSCREEN);
    cy := GetSystemMetrics(SM_CYSCREEN);
    GetWindowRect(Wnd, LRect);
    w := LRect.Right - LRect.Left;
    h := LRect.Bottom - LRect.Top;
    x := (cx - w) div 2;
    y := (cy - h) div 2;
    SetWindowPos(Wnd, 0, x, y, 0, 0, SWP_NOSIZE or SWP_NOZORDER);
  end;
  Result := 0;
end;

function SelectDirectoryEx(const Caption: string; const Root: WideString; var Directory: string): Boolean;
{$IFDEF VER140} // Delphi 6
const
  BIF_NEWDIALOGSTYLE = $0040;
  BIF_USENEWUI = BIF_NEWDIALOGSTYLE or BIF_EDITBOX;
{$ENDIF}
var
  WindowList: Pointer;
  BrowseInfo: TBrowseInfo;
  Buffer: PChar;
  OldErrorMode: Cardinal;
  RootItemIDList, ItemIDList: PItemIDList;
  ShellMalloc: IMalloc;
  IDesktopFolder: IShellFolder;
  Eaten, Flags: LongWord;
begin
  Result := False;
  if not DirectoryExists(Directory) then
    Directory := '';
  Directory := ExpandFileName(Directory);
  FillChar(BrowseInfo, SizeOf(BrowseInfo), 0);
  if (ShGetMalloc(ShellMalloc) = S_OK) and (ShellMalloc <> nil) then
  begin
    Buffer := ShellMalloc.Alloc(MAX_PATH);
    try
      RootItemIDList := nil;
      if Root <> '' then
      begin
        SHGetDesktopFolder(IDesktopFolder);
        IDesktopFolder.ParseDisplayName(Application.Handle, nil,
          POleStr(Root), Eaten, RootItemIDList, Flags);
      end;
      with BrowseInfo do
      begin
        hwndOwner := Application.Handle;
        pidlRoot := RootItemIDList;
        pszDisplayName := Buffer;
        lpszTitle := PChar(Caption);
        ulFlags := BIF_RETURNONLYFSDIRS or BIF_DONTGOBELOWDOMAIN or BIF_USENEWUI;
        lpfn := SelectDirCBEx;
        if Directory <> '' then
          lParam := NativeInt(PChar(Directory));
      end;
      WindowList := DisableTaskWindows(0);
      OldErrorMode := SetErrorMode(SEM_FAILCRITICALERRORS);
      try
        ItemIDList := ShBrowseForFolder(BrowseInfo);
      finally
        SetErrorMode(OldErrorMode);
        EnableTaskWindows(WindowList);
      end;
      Result := ItemIDList <> nil;
      if Result then
      begin
        ShGetPathFromIDList(ItemIDList, Buffer);
        ShellMalloc.Free(ItemIDList);
        if Buffer <> '' then
          Directory := IncludeTrailingPathDelimiter(Buffer)
        else
          Result := False;
      end;
    finally
      ShellMalloc.Free(Buffer);
    end;
  end;
end;

{ TfrmOption }

procedure TfrmOption.FormCreate(Sender: TObject);
begin
  Image1.Hint := SViewActual;
  Image1.ShowHint := True;
  edtOutPath.Text := SSameAsInput;
  FBitmap := TBitmap.Create;
  ClearFrameView(True);
end;

procedure TfrmOption.FormShow(Sender: TObject);
begin
  if grpFormats.CanFocus and (grpFormats.ItemIndex >= 0) then
{$IFDEF VER140} // Delphi 6
    grpFormats.SetFocus;
{$ELSE}
    grpFormats.Buttons[grpFormats.ItemIndex].SetFocus;
{$ENDIF}
end;

procedure TfrmOption.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
var
  LDriveType: UINT;
begin
  if Self.ModalResult = mrOk then
  begin
    if edtOutPath.Text = SSameAsInput then
      LDriveType := GetDriveType(PChar(ExtractFileDrive(FDecoder.FileName)))
    else
      LDriveType := GetDriveType(PChar(ExtractFileDrive(edtOutPath.Text)));

    if LDriveType = DRIVE_CDROM then
    begin
      Application.MessageBox('Input folder is located in CD-ROM, please select output folder.',
        PChar(Application.Title), MB_ICONINFORMATION);
      btnOutPath.SetFocus;
      CanClose := False;
    end;
  end;
end;

procedure TfrmOption.FormDestroy(Sender: TObject);
begin
  FBitmap.Free;
end;

procedure TfrmOption.ClearFrameView(AResetCutClip: Boolean);
begin
  FCurrentPTS := 0;
  lblPTS.Caption := DurationToStr(0);
  // clear cached frame picture
  FBitmap.Width := 0;
  FBitmap.Height := 0;
  // fill background
  Image1.Canvas.Brush.Color := clSkyBlue;
  Image1.Canvas.FillRect(Image1.Canvas.ClipRect);
  // diable track bar
  TrackBar1.Enabled := False;
  TrackBar1.Position := 0;
  SetupTrackBar;
  if AResetCutClip then
  begin
    // reset global variables
    FDuration := 0;
    FFrames := 0;
    FFirstPTS := High(Int64);
    FStartTime := High(Int64);
    FEndTime := AV_NOPTS_VALUE;
    // adjust cut clip controls
    TrackBar1.SelStart := 0;
    TrackBar1.SelEnd := 0;
    chkCutClip.Enabled := False;
    chkCutClip.Checked := False;
    edtStartTime.Text := 'N/A';
    edtEndTime.Text := 'N/A';
  end;
end;

procedure TfrmOption.SetupTrackBar;
var
  LEnabled: Boolean;
  LPosRate: Double;
begin
  LEnabled := TrackBar1.Enabled;

  TrackBar1.Enabled := False;

  if TrackBar1.Max > 0 then
    LPosRate := TrackBar1.Position / TrackBar1.Max
  else
    LPosRate := 0;

  if FFrames > 0 then
{$IF CompilerVersion >= 20} // Delphi 2009 or above
    TrackBar1.Max := FFrames;
{$ELSE}
    if FFrames > $7FFF then
      TrackBar1.Max := $7FFF
    else
      TrackBar1.Max := FFrames;
{$IFEND}

  if TrackBar1.Max > 10 then
    TrackBar1.Frequency := TrackBar1.Max div 10
  else
    TrackBar1.Frequency := 1;

  if TrackBar1.Max < 10 then
    TrackBar1.PageSize := 2
  else if TrackBar1.Max < 100 then
    TrackBar1.PageSize := 5
  else
    TrackBar1.PageSize := TrackBar1.Max div 20;

  TrackBar1.Position := Round(TrackBar1.Max * LPosRate);
  FLastPosition := TrackBar1.Position;

  TrackBar1.Enabled := LEnabled;
  TrackBar1.SliderVisible := LEnabled;
end;

procedure TfrmOption.DrawCurrentFrame;
var
  LPosition: Integer;
begin
  // update current pts and current position
  FCurrentPTS := FDecoder.FrameInfo.PTS;
  if FFirstPTS = High(Int64) then
    FFirstPTS := FCurrentPTS;
  lblPTS.Caption := DurationToStr(FCurrentPTS);
  if not FChanging and (FDuration > 0) then
  begin
    FChanging := True;
    try
      LPosition := Round(TrackBar1.Max * FCurrentPTS / FDuration);
      if LPosition > TrackBar1.Max then
        LPosition := TrackBar1.Max;
      TrackBar1.Position := LPosition;
    finally
      FChanging := False;
    end;
  end;
  // copy the frame to bitmap
  if FDecoder.CopyToBitmap(FBitmap) then
    FitDraw(Image1.Canvas, FBitmap)
  else
    //mmoLog.Lines.Add('***CopyToBitmap failed: ' + FDecoder.LastErrMsg);
    ;
end;

procedure TfrmOption.ReadAndDrawNextFrame;
begin
  // next frame
  if FDecoder.Decode then
    DrawCurrentFrame
  else
    //mmoLog.Lines.Add('***Decode failed: ' + FDecoder.LastErrMsg);
    ;
end;

procedure TfrmOption.chkAudioVolumeClick(Sender: TObject);
begin
  edtAudioVolume.Enabled := chkAudioVolume.Checked;
end;

procedure TfrmOption.chkCropClick(Sender: TObject);
begin
  edtCropLeft.Enabled := chkCrop.Checked;
  edtCropRight.Enabled := chkCrop.Checked;
  edtCropTop.Enabled := chkCrop.Checked;
  edtCropBottom.Enabled := chkCrop.Checked;
end;

procedure TfrmOption.chkPadClick(Sender: TObject);
begin
  edtPadLeft.Enabled := chkPad.Checked;
  edtPadRight.Enabled := chkPad.Checked;
  edtPadTop.Enabled := chkPad.Checked;
  edtPadBottom.Enabled := chkPad.Checked;
end;

procedure TfrmOption.chkRotateClick(Sender: TObject);
begin
  edtRotate.Enabled := chkRotate.Checked;
end;

procedure TfrmOption.chkBrightnessClick(Sender: TObject);
begin
  trbBrightness.Enabled := chkBrightness.Checked;
  btnBrightness.Enabled := chkBrightness.Checked;
end;

procedure TfrmOption.chkSaturationClick(Sender: TObject);
begin
  trbSaturation.Enabled := chkSaturation.Checked;
  btnSaturation.Enabled := chkSaturation.Checked;
end;

procedure TfrmOption.chkHueClick(Sender: TObject);
begin
  trbHue.Enabled := chkHue.Checked;
  btnHue.Enabled := chkHue.Checked;
end;

procedure TfrmOption.btnOutPathClick(Sender: TObject);
var
  LPath: string;
begin
  if edtOutPath.Text = SSameAsInput then
    LPath := ExtractFilePath(FDecoder.FileName)
  else
    LPath := edtOutPath.Text;
  if SelectDirectoryEx('Select Output Folder', '', LPath) then
    if GetDriveType(PChar(ExtractFileDrive(LPath))) = DRIVE_CDROM then
      Application.MessageBox('Output folder cannot be located in CD-ROM.',
        PChar(Application.Title), MB_ICONINFORMATION)
    else
      edtOutPath.Text := LPath;
end;

procedure TfrmOption.DoActualFormClick(Sender: TObject);
begin
  if Assigned(Sender) and (Sender is TControl) and ((Sender as TControl).Parent is TForm) then
    ((Sender as TControl).Parent as TForm).Close;
end;

procedure TfrmOption.Image1Click(Sender: TObject);
var
  F: TForm;
  I: TImage;
  B: TButton;
begin
  // sanity check to ensure the current frame is available
  if (FBitmap.Width = 0) or (FBitmap.Height = 0) then
    Exit;

  // show current frame picture in actual size
  F := TForm.Create(Self);
  try
    // create a Button to response Enter and ESC key-press
    B := TButton.Create(F);
    with B do
    begin
      Parent := F;
      Default := True;
      Cancel := True;
      ModalResult := mrCancel;
      Top := 0;
      Left := 0;
      Width := 0;
      Height := 0;
      Visible := True;
    end;
    // create an Image to display the frame
    I := TImage.Create(F);
    with I do
    begin
      Parent := F;
      OnClick := Self.DoActualFormClick;
      Visible := True;
      Left := 0;
      Top := 0;
      Width := FBitmap.Width;
      Height := FBitmap.Height;
      Picture.Assign(FBitmap);
    end;
    // setup the form and then show it
    with F do
    begin
      BorderIcons := [biSystemMenu];
      BorderStyle := bsSingle;
      Position := poScreenCenter;
      Caption := Format('%dx%d @ %s', [FBitmap.Width, FBitmap.Height,
        DurationToStr(FCurrentPTS)]);;
      ClientWidth := I.Width;
      ClientHeight := I.Height;
      ShowModal;
    end;
  finally
    F.Free;
  end;
end;

procedure TfrmOption.TrackBar1Change(Sender: TObject);
var
  LDesirePTS: Int64;
begin
  if FDecoder.VideoStreamCount <= 0 then
  begin
    // if no video stream, that is, only audio stream
    // update current pts and current position
    if TrackBar1.Max > 0 then
      FCurrentPTS := Round(TrackBar1.Position / TrackBar1.Max * FDuration)
    else
      FCurrentPTS := 0;
    lblPTS.Caption := DurationToStr(FCurrentPTS);
    Exit;
  end;
  // check the situation for seeking
  if FChanging or not TrackBar1.Enabled or (FLastPosition = TrackBar1.Position) then
    Exit;

  FChanging := True;
  try
    // position rate
    if TrackBar1.Max > 0 then
      LDesirePTS := Round(TrackBar1.Position / TrackBar1.Max * FDuration)
    else
      LDesirePTS := 0;

    if TrackBar1.Position = FLastPosition + 1 then
    begin
      // next frame
      if FDecoder.Decode then
        DrawCurrentFrame
      else
        //mmoLog.Lines.Add('***Decode failed: ' + FDecoder.LastErrMsg);
        ;
    end
    else if TrackBar1.Position = FLastPosition - 1 then
    begin
      if FDecoder.DecodePreviousFrame then
        DrawCurrentFrame
      else
        //mmoLog.Lines.Add('***Decode failed: ' + FDecoder.LastErrMsg);
        ;
    end
    else
    begin
      // try to seek the frame
      if FDecoder.Seek(LDesirePTS) or
        FDecoder.Seek(LDesirePTS, [sfBackward]) or
        FDecoder.Seek(LDesirePTS, [sfAny]) then
      begin
        // decode the next frame
        while FDecoder.Decode do
        begin
          // if the frame is the one we desire
          if (FDecoder.FrameInfo.PTS >= LDesirePTS) or (LDesirePTS <= 0) then
          begin
            DrawCurrentFrame;
            Exit;
          end;
        end;
        //mmoLog.Lines.Add('***Decode failed: ' + FDecoder.LastErrMsg);
      end
      else
        //mmoLog.Lines.Add('***SeekFrame failed: ' + FDecoder.LastErrMsg);
        ;
    end;
  finally
    FLastPosition := TrackBar1.Position;
    FChanging := False;
  end;
end;

procedure TfrmOption.btnStartTimeClick(Sender: TObject);
begin
  TrackBar1.SelStart := TrackBar1.Position;
  edtStartTime.Text := DurationToStr(FCurrentPTS);
  FStartTime := FCurrentPTS;

  chkCutClip.Enabled := FEndTime > FStartTime;
end;

procedure TfrmOption.btnEndTimeClick(Sender: TObject);
begin
  TrackBar1.SelEnd := TrackBar1.Position;
  edtEndTime.Text := DurationToStr(FCurrentPTS);
  FEndTime := FCurrentPTS;

  chkCutClip.Enabled := FEndTime > FStartTime;
end;

procedure TfrmOption.btnNextFrameClick(Sender: TObject);
begin
  ReadAndDrawNextFrame;
end;

procedure TfrmOption.btnBrightnessClick(Sender: TObject);
begin
  trbBrightness.Position := 0;
end;

procedure TfrmOption.btnSaturationClick(Sender: TObject);
begin
  trbSaturation.Position := 10;
end;

procedure TfrmOption.btnHueClick(Sender: TObject);
begin
  trbHue.Position := 0;
end;

procedure TfrmOption.SetDecoder(const Value: TFFDecoder);
begin
  // clear frame view and reset global variables
  ClearFrameView(not Assigned(FDecoder) or
    (FDecoder.FileName <> Value.FileName) or
    (FDecoder.FileStreamInfo.Duration <> Value.FileStreamInfo.Duration) or
    (FDecoder.FileStreamInfo.StartTime <> Value.FileStreamInfo.StartTime));

  FDecoder := Value;

  // display AVFileInfo
  mmoAVFileInfo.Text := FDecoder.FileInfoText;

  // NOTICE: this demo application only shows first video stream and/or first audio stream.
  FVideoStreamIndex := FDecoder.FirstVideoStreamIndex;
  FAudioStreamIndex := FDecoder.FirstAudioStreamIndex;

  with FDecoder do
  begin
    // whether we can seek or not
    TrackBar1.Enabled := (FileStreamInfo.Duration <> AV_NOPTS_VALUE) and (FormatName <> 'image2');

    // file duration
    if FileStreamInfo.Duration <> AV_NOPTS_VALUE then
      FDuration := FileStreamInfo.Duration;

    // calculate Frames of the first video stream, use it for the track bar max position
    if (VideoStreamCount > 0) and (FirstVideoStreamInfo.FrameRate.den > 0) then
    begin
      if FirstVideoStreamInfo.DurationScaled > 0 then
        FFrames := Round(FirstVideoStreamInfo.DurationScaled / AV_TIME_BASE *
          FirstVideoStreamInfo.FrameRate.num / FirstVideoStreamInfo.FrameRate.den)
      else if FileStreamInfo.Duration <> AV_NOPTS_VALUE then
        FFrames := Round(FileStreamInfo.Duration / AV_TIME_BASE *
          FirstVideoStreamInfo.FrameRate.num / FirstVideoStreamInfo.FrameRate.den);
    end
    else if FileStreamInfo.Duration <> AV_NOPTS_VALUE then
      FFrames := Round(FileStreamInfo.Duration / AV_TIME_BASE * 10);
  end;

  // if video stream available, then read and draw the first frame
  if FDecoder.VideoStreamCount > 0 then
    ReadAndDrawNextFrame;

  // setup track bar for seeking
  SetupTrackBar;

  // TODO: here you can show more special AVFileInfo in detail and
  //  initialize preset options according the input file
end;

{******************************************
          time line
  |-----------------------------|
  ^     ^            ^          ^
 zero   |            |       total duration
      StartTime    EndTime
      (offset)     (offset)
        |------------|
         \          /
          TimeLength
        (new duration)
******************************************}

function TfrmOption.GetClipStartTime: Int64;
begin
  // start time offset
  if (FFirstPTS <> High(Int64)) and (FFirstPTS < 0) then
    Result := FStartTime - FFirstPTS
  else
    Result := FStartTime;
end;

function TfrmOption.GetClipTimeLength: Int64;
begin
  // new duration of the new piece clip
  Result := FEndTime - FStartTime;
end;

function GetCPUs: DWORD;
var
  LSystemInfo: TSystemInfo;
begin
  GetSystemInfo(LSystemInfo);
  Result := LSystemInfo.dwNumberOfProcessors;
end;

function TfrmOption.GetEventOptions: TEventOptions;
begin
  InitEventOptions(@Result);
  // whether enable OnAudioOutputHook event
  Result.AudioOutputHook := chkAudioHook.Checked;
  // whether enable OnVideoInputHook event
  Result.VideoInputHook := chkVideoInputHook.Checked;
  // whether enable OnVideoOutputHook event
  Result.VideoOutputHook := chkVideoOutputHook.Checked;
end;

function TfrmOption.GetOptions: string;
  function BuildCropFilter: string;
  var
    W, H, T, B, L, R: Integer;
  begin
    // libavfilter: please refer to https://ffmpeg.org/ffmpeg-filters.html
    // NOTICE: What I know about libavfilter is also very limited.
    //         so you have to get help of it by yourself.
    W := FDecoder.VideoStreamInfos[FVideoStreamIndex].Width;
    H := FDecoder.VideoStreamInfos[FVideoStreamIndex].Height;

    T := StrToIntDef(edtCropTop.Text, 0);
    B := StrToIntDef(edtCropBottom.Text, 0);
    L := StrToIntDef(edtCropLeft.Text, 0);
    R := StrToIntDef(edtCropRight.Text, 0);

    if (T > 0) or (B > 0) or (L > 0) or (R > 0) then
      // Crop the input video to out_w:out_h:x:y.
      Result := Format('crop=%d:%d:%d:%d', [W - L - R, H - T - B, L, T])
    else
      Result := '';
  end;

  function BuildPadFilter: string;
  var
    W, H, T, B, L, R: Integer;
  begin
    // libavfilter: please refer to https://ffmpeg.org/ffmpeg-filters.html
    // NOTICE: What I know about libavfilter is also very limited.
    //         so you have to get help of it by yourself.
    W := FDecoder.VideoStreamInfos[FVideoStreamIndex].Width;
    H := FDecoder.VideoStreamInfos[FVideoStreamIndex].Height;

    // fix frame size after crop filter
    if chkCrop.Checked and (FVideoStreamIndex >= 0) then
    begin
      T := StrToIntDef(edtCropTop.Text, 0);
      B := StrToIntDef(edtCropBottom.Text, 0);
      L := StrToIntDef(edtCropLeft.Text, 0);
      R := StrToIntDef(edtCropRight.Text, 0);
      if (T > 0) or (B > 0) or (L > 0) or (R > 0) then
      begin
        W := W - L - R;
        H := H - T - B;
      end;
    end;

    T := StrToIntDef(edtPadTop.Text, 0) div 2 * 2;
    B := StrToIntDef(edtPadBottom.Text, 0) div 2 * 2;
    L := StrToIntDef(edtPadLeft.Text, 0) div 2 * 2;
    R := StrToIntDef(edtPadRight.Text, 0) div 2 * 2;

    if (T > 0) or (B > 0) or (L > 0) or (R > 0) then
      // pad filter accepts the following parameters: width:height:x:y:color.
      Result := Format('pad=%d:%d:%d:%d:blue', [W + L + R, H + T + B, L, T])
    else
      Result := '';
  end;

  function BuildWaterMarkFilter(AOtherFilters: string): string;
  var
    LPngFile: string;
    LWaterMark: string;
    LOverlay: string;
  begin
    // libavfilter: please refer to https://ffmpeg.org/ffmpeg-filters.html
    // NOTICE: What I know about libavfilter is also very limited.
    //         so you have to get help of it by yourself.
    LPngFile := ExtractFilePath(Application.ExeName) + CWaterMark;
    LPngFile := StringReplace(LPngFile, '\', '/', [rfReplaceAll]);
    LPngFile := StringReplace(LPngFile, ':', '\\:', [rfReplaceAll]);
    LWaterMark := Format('movie=%s [watermark]', [LPngFile]);
    LOverlay := '[watermark] overlay=main_w-overlay_w-10:10';
    if AOtherFilters <> '' then
      // other filters effect overlay
      //Result := LWaterMark + '; [in] ' + LOverlay + ', ' + AOtherFilters + ' [out]'
      // other filters do not effect overlay
      Result := LWaterMark + '; [in] ' + AOtherFilters + ' [main]; [main] ' + LOverlay + ' [out]'
    else
      Result := LWaterMark + '; [in] ' + LOverlay + ' [out]';
  end;
var
  LOptions: string;
  LVideoFilters: string;
  procedure AppendVideoFilter(ANewFilter: string);
  begin
    if ANewFilter <> '' then
    begin
      if LVideoFilters <> '' then
        LVideoFilters := LVideoFilters + ',' + ANewFilter
      else
        LVideoFilters := ANewFilter;
    end;
  end;

  procedure AppendOption(AOption: string);
  begin
    LOptions := LOptions + ' ' + AOption;
  end;
var
  LAudioChannels: string;
  LAudioSampleRate: string;
  LFrameSize: string;
  LFrameRate: string;
  LAudioCodec: string;
  LVideoCodec: string;
  LTemp: string;
  LFileExt: string;
  LBasename: string;
  LFilename: string;
  I: Integer;
begin
  LOptions := '';
  LVideoFilters := '';

  LAudioChannels := '';
  LAudioSampleRate := '';
  LFrameSize := '';
  LFrameRate := '';
  LAudioCodec := '';
  LVideoCodec := '';


// **********Options for input*************
{$IFDEF QUICK_CUT}
  // cut a piece clip, see also -t in OutputOptions
  if chkCutClip.Enabled and chkCutClip.Checked and (FEndTime > FStartTime) then
    AppendOption('-ss ' + FloatToStr(GetClipStartTime / 1000000)); // start time offset
{$ENDIF}
  if Pos(LowerCase(grpFormats.Items.Strings[grpFormats.ItemIndex]),
    'aac ac3 mp2 mp3 wav wma') > 0 then
    AppendOption('-vn'); // disable input video
  AppendOption(Format('-i "%s"', [FDecoder.FileName]));


// **********Options for output*************
  // Audio Channels
  if grpAudioChannels.ItemIndex > 0 then
    LAudioChannels := IntToStr(grpAudioChannels.ItemIndex);

  // Audio Sample Rate: Hz value
  if grpAudioSampleRate.ItemIndex > 0 then
    LAudioSampleRate := grpAudioSampleRate.Items.Strings[grpAudioSampleRate.ItemIndex];

  // Video Frame Size: WxH or abbreviation
  if cboFrameSize.ItemIndex > 0 then
  begin
    LTemp := cboFrameSize.Items.Strings[cboFrameSize.ItemIndex];
    if Pos(' ', LTemp) > 1 then
      LTemp := Copy(LTemp, 1, Pos(' ', LTemp) - 1);
    LFrameSize := LTemp;
  end;

  // Frame Rate(FPS): Hz value, fraction or abbreviation
  if grpFrameRate.ItemIndex > 0 then
    LFrameRate := grpFrameRate.Items.Strings[grpFrameRate.ItemIndex];

  // Audio Codec(Encoder)
  if grpAudioCodec.ItemIndex > 0 then
    LAudioCodec := LowerCase(grpAudioCodec.Items.Strings[grpAudioCodec.ItemIndex]);

  // Video Codec(Encoder)
  if grpVideoCodec.ItemIndex > 0 then
    LVideoCodec := LowerCase(grpVideoCodec.Items.Strings[grpVideoCodec.ItemIndex]);

  // Audio Bit Rate: in bits/s
  AppendOption('-b:a ' + edtAudioBitRate.Text + 'k');

  // Video Bit Rate: in bits/s
  AppendOption('-b:v ' + edtVideoBitRate.Text + 'k');

  // Aspect Ratio: 4:3, 16:9 or 1.3333, 1.7777
  if grpAspectRatio.ItemIndex > 0 then
    AppendOption('-aspect ' + grpAspectRatio.Items.Strings[grpAspectRatio.ItemIndex]);

{$IFNDEF QUICK_CUT}
  // cut a piece clip
  if chkCutClip.Enabled and chkCutClip.Checked and (FEndTime > FStartTime) then
    AppendOption('-ss ' + FloatToStr(GetClipStartTime / 1000000)); // start time offset
{$ENDIF}

  // cut a piece clip, see also -ss in InputOptions/OutputOptions
  if chkCutClip.Enabled and chkCutClip.Checked and (FEndTime > FStartTime) then
    AppendOption('-t ' + FloatToStr(GetClipTimeLength / 1000000)); // duration of the new piece clip

  // here we use FileExt to indicate the output format
  LFileExt := '.' + LowerCase(grpFormats.Items.Strings[grpFormats.ItemIndex]);

  // special options depend on output formats
  if (LFileExt = '.flv') or (LFileExt = '.swf') then
  begin
    // FLV/SWF only supports this three sample rate, (44100, 22050, 11025).
    if (LAudioSampleRate <> '11025') and (LAudioSampleRate <> '22050') and (LAudioSampleRate <> '44100') then
      LAudioSampleRate := '44100';
  end
  else if LFileExt = '.ogg' then
  begin
    LAudioCodec := 'libvorbis';
    // Ogg with video
    LVideoCodec := 'libtheora';
    // Ogg Vorbis Audio only
    //AppendOption('-vn');
  end
  else if LFileExt = '.psp' then
  begin
(*
How do I encode videos which play on the PSP?
`needed stuff'
	-acodec aac -vcodec mpeg4 width*height<=76800 width%16=0 height%16=0 -ar 24000 -r 30000/1001 or 15000/1001 -f psp
`working stuff'
	4mv, title
`non-working stuff'
	B-frames
`example command line'
	ffmpeg -i input -acodec aac -ab 128kb -vcodec mpeg4 -b 1200kb -ar 24000 -mbd 2 -flags +4mv -trellis 2 -aic 2 -cmp 2 -subcmp 2 -s 368x192 -r 30000/1001 -metadata title=X -f psp output.mp4
`needed stuff for H.264'
	-acodec aac -vcodec libx264 width*height<=76800 width%16=0? height%16=0? -ar 48000 -coder 1 -r 30000/1001 or 15000/1001 -f psp
`working stuff for H.264'
	title, loop filter
`non-working stuff for H.264'
	CAVLC
`example command line'
	ffmpeg -i input -acodec aac -ab 128kb -vcodec libx264 -b 1200kb -ar 48000 -mbd 2 -coder 1 -cmp 2 -subcmp 2 -s 368x192 -r 30000/1001 -metadata title=X -f psp -flags loop -trellis 2 -partitions parti4x4+parti8x8+partp4x4+partp8x8+partb8x8 output.mp4
`higher resolution for newer PSP firmwares, width<=480, height<=272'
	-vcodec libx264 -level 21 -coder 1 -f psp
`example command line'
	ffmpeg -i input -acodec aac -ab 128kb -ac 2 -ar 48000 -vcodec libx264 -level 21 -b 640kb -coder 1 -f psp -flags +loop -trellis 2 -partitions +parti4x4+parti8x8+partp4x4+partp8x8+partb8x8 -g 250 -s 480x272 output.mp4
*)
    AppendOption('-f psp');
    LVideoCodec := 'mpeg4';
    LAudioCodec := 'aac';
    LFrameSize := '320x240';
    LAudioSampleRate := '24000';
    LFrameRate := '15000/1001';
    LFileExt := '_psp.mp4';
  end
  else if LFileExt = '.ipod' then
  begin
(*
How do I encode videos which play on the iPod?
`needed stuff'
	-acodec aac -vcodec mpeg4 width<=320 height<=240
`working stuff'
	4mv, title
`non-working stuff'
	B-frames
`example command line'
	ffmpeg -i input -acodec aac -ab 128kb -vcodec mpeg4 -b 1200kb -mbd 2 -flags +4mv -trellis 2 -aic 2 -cmp 2 -subcmp 2 -s 320x180 -metadata title=X output.mp4
*)
    LVideoCodec := 'mpeg4';
    LAudioCodec := 'aac';
    LFrameSize := '320x240';
    if LFrameRate = '' then
      LFrameRate := '15000/1001';
    LFileExt := '_ipod.mp4';
  end
  else if LFileExt = '.rm10' then
  begin
    LVideoCodec := 'rv10';
    LFileExt := '_rv10.rm';
  end
  else if LFileExt = '.rm20' then
  begin
    LVideoCodec := 'rv20';
    LFileExt := '_rv20.rm';
  end
  else if LFileExt = '.webm' then
  begin
    AppendOption('-f webm');
    LVideoCodec := 'libvpx';
    LAudioCodec := 'libvorbis';
  end
  else if LFileExt = '.wmv' then
  begin
    LVideoCodec := 'wmv2';
    LAudioCodec := 'wmav2';
  end
  else if LFileExt = '.mkv' then
  begin
   LVideoCodec := 'libx264';
    //LAudioCodec := 'ac3';
  end
  else if LFileExt = '.wma' then
  begin
    AppendOption('-vn');
    LAudioCodec := 'wmav2';
  end
  else if LFileExt = '.3gp' then
  begin
    // Only mono supported
    LAudioChannels := '1';
    // Only 8000Hz sample rate supported
    LAudioSampleRate := '8000';
    // bitrate: use one of 4.75k, 5.15k, 5.9k, 6.7k, 7.4k, 7.95k, 10.2k or 12.2k
(*
    AMR_bitrates rates[]={ {4750,MR475},
                           {5150,MR515},
                           {5900,MR59},
                           {6700,MR67},
                           {7400,MR74},
                           {7950,MR795},
                           {10200,MR102},
                           {12200,MR122},
                         };
*)
    AppendOption('-b:a 12.2k');
    // [h263 @ 6BF9AE00]The specified picture size of 320x240 is not valid for the H.263 codec.
    //    Valid sizes are 128x96, 176x144, 352x288, 704x576, and 1408x1152. Try H.263+.
    LFrameSize := '128x96';
  end
  else if Pos(LFileExt, '.vcd.svcd.dvd') > 0 then
  begin
    if LFileExt = '.vcd' then
      LTemp := 'vcd'
    else if LFileExt = '.svcd' then
      LTemp := 'svcd'
    else // if Ext = '.dvd' then
      LTemp := 'dvd';

    if grpNorm.ItemIndex = 1 then
      LTemp := 'pal-' + LTemp
    else if grpNorm.ItemIndex = 2 then
      LTemp := 'ntsc-' + LTemp;

    AppendOption('-target ' + LTemp);
  end;

  // -ac channels        set number of audio channels
  if LAudioChannels <> '' then
    AppendOption('-ac ' + LAudioChannels);
  // -ar rate            set audio sampling rate (in Hz)
  if LAudioSampleRate <> '' then
    AppendOption('-ar ' + LAudioSampleRate);
  // -r rate             set frame rate (Hz value, fraction or abbreviation)
  if LFrameRate <> '' then
    AppendOption('-r ' + LFrameRate);

  // -c codec            codec name
  // audio codec
  if LAudioCodec = 'mp3' then
    LAudioCodec := 'libmp3lame';
  if LAudioCodec <> '' then
    AppendOption('-c:a ' + LAudioCodec);
  // video codec
  if LVideoCodec = 'xvid' then
    LVideoCodec := 'libxvid';
  if LVideoCodec = 'x264' then
    LVideoCodec := 'libx264';
  if LVideoCodec = 'qsv' then
    LVideoCodec := 'h264_qsv';
  if LVideoCodec = 'divx' then
  begin
    LVideoCodec := '';
    AppendOption('-c:v mpeg4 -tag:v DIVX -vstrict unofficial');
    if LFrameRate = '' then
      AppendOption('-r 15000/1001');
  end;
  if LVideoCodec <> '' then
    AppendOption('-c:v ' + LVideoCodec);

{
  // MP3 muxer AVOptions:
  // -id3v2_version     <int>   E.... Select ID3v2 version to write. Currently 3 and 4 are supported.
  // -write_id3v1       <int>   E.... Enable ID3v1 writing. ID3v1 tags are written in UTF-8 which may not be supported by most software.

  // -metadata string=string  add metadata

  // ID3v1
  AppendOption('-metadata track="1"');
  AppendOption('-metadata title="FFVCL title"');
  AppendOption('-metadata artist="FFVCL artist"');
  AppendOption('-metadata album="FFVCL album"');
  AppendOption('-metadata genre="Dance Hall"');
  AppendOption('-metadata comment="FFVCL comment"');
  AppendOption('-metadata TDRL=2013'); // year

  // ID3v1 enable
  AppendOption('-write_id3v1 1');

  // ID3v2
  AppendOption('-metadata album_artist="FFVCL album artist"');
  AppendOption('-metadata performer="FFVCL performer"');
  AppendOption('-metadata encoded_by="FFVCL encoded_by"');
  AppendOption('-metadata copyright="FFVCL copyright"');
  AppendOption('-metadata composer="FFVCL composer"');
  AppendOption('-metadata TOPE="FFVCL Original Artist"');
  AppendOption('-metadata TYER="2013"'); // year
  AppendOption('-metadata publisher="FFVCL publisher"');
  AppendOption('-metadata encoder="FFVCL encoder"');
  AppendOption('-metadata disc="FFVCL disc"');
  AppendOption('-metadata language=eng');

  // ID3v2 version 3 or 4
  AppendOption('-id3v2_version 3');
}

  // **********The following code shows Video Filter usage*************

  // libavfilter: please refer to https://ffmpeg.org/ffmpeg-filters.html
  // NOTICE: What I know about libavfilter is also very limited.
  //         so you have to get help of it by yourself.

  // TODO: filter Audio Volume
  if chkAudioVolume.Checked then
    AppendOption('-af volume=' + edtAudioVolume.Text);

  // Crop the input video to out_w:out_h:x:y.
  if chkCrop.Checked and (FVideoStreamIndex >= 0) then
    AppendVideoFilter(BuildCropFilter);

  // Add paddings to the input image, and places the original input at the given coordinates x, y.
  if chkPad.Checked and (FVideoStreamIndex >= 0) then
    AppendVideoFilter(BuildPadFilter);

  // Scale the input video to width:height and/or convert the image format.
  If (LFrameSize <> '') And (FVideoStreamIndex >= 0) Then
    AppendVideoFilter('scale=' + StringReplace(LFrameSize, 'x', ':', []));

  // vflip: Vertically Flip Video Frame
  if chkVertFlip.Checked then
    AppendVideoFilter('vflip');

  // hflip: Horizontally Flip Video Frame
  if chkHoriFlip.Checked then
    AppendVideoFilter('hflip');

  // yadif: Deinterlace the input image
  if chkDeinterlace.Checked then
    AppendVideoFilter('yadif');
    //AppendOption('-deinterlace');

  // negate: Negate Video Frame
  if chkNegate.Checked then
    AppendVideoFilter('negate');

  // rotate: Rotate video by an arbitrary angle expressed in radians.
  if chkRotate.Checked then
    AppendVideoFilter('rotate=' + FloatToStr(StrToIntDef(Trim(edtRotate.Text), 45) / 180 * PI));
    //AppendVideoFilter('rotate=' + FloatToStr(StrToIntDef(Trim(edtRotate.Text), 45) / 180 * PI) + ':ow=hypot(iw\,ih):oh=ow');

  // hue: Modify the hue and/or the saturation of the input.
  if chkBrightness.Checked or chkSaturation.Checked or chkHue.Checked then
  begin
    LTemp := '';
    if chkBrightness.Checked then
      LTemp := LTemp + ':b=' + FloatToStr(trbBrightness.Position / 10);
    if chkSaturation.Checked then
      LTemp := LTemp + ':s=' + FloatToStr(trbSaturation.Position / 10);
    if chkHue.Checked then
      LTemp := LTemp + ':H=' + FloatToStr(trbHue.Position / 100 * PI);
    AppendVideoFilter('hue=' + Copy(LTemp, 2, MaxInt));
  end;

  // watermark: image overlay
  if chkWaterMark.Checked then
    LVideoFilters := BuildWaterMarkFilter(LVideoFilters);

  // -vf filter list     video filters
  if LVideoFilters <> '' then
    AppendOption('-vf "' + LVideoFilters + '"');

  // encoding in multiple threads depends on encoder capability (such as libx264 which supports threads)
  // if the ffmpeg libraries were not compiled with thread support, it would be using thread emulation
  if (LVideoCodec = 'libx264') and (GetCPUs > 1) then
    AppendOption('-threads ' + IntToStr(GetCPUs));
  if LVideoCodec = 'libx264' then
    AppendOption('-pix_fmt yuv420p');

  if chkAudioHook.Checked then
  begin
{
aac -> fltp
ac3 -> fltp
ac3_fixed -> s16p
adpcm_adx -> s16
adpcm_ima_qt -> s16p
adpcm_ima_wav -> s16p
adpcm_ms -> s16
adpcm_swf -> s16
adpcm_yamaha -> s16
alac -> s32p, s16p
comfortnoise -> s16
dca -> s32
eac3 -> fltp
flac -> s16, s32
g722 -> s16
g723_1 -> s16
g726 -> s16
libgsm -> s16
libgsm_ms -> s16
libilbc -> s16
libmp3lame -> s32p, fltp, s16p
libopencore_amrnb -> s16
libopus -> s16, flt
libspeex -> s16
libtwolame -> flt, fltp, s16, s16p
libvo_aacenc -> s16
libvo_amrwbenc -> s16
libvorbis -> fltp
libwavpack -> s32
mp2 -> s16
nellymoser -> flt
pcm_alaw -> s16
pcm_f32be -> flt
pcm_f32le -> flt
pcm_f64be -> dbl
pcm_f64le -> dbl
pcm_mulaw -> s16
pcm_s16be -> s16
pcm_s16be_planar -> s16p
pcm_s16le -> s16
pcm_s16le_planar -> s16p
pcm_s24be -> s32
pcm_s24daud -> s16
pcm_s24le -> s32
pcm_s24le_planar -> s32p
pcm_s32be -> s32
pcm_s32le -> s32
pcm_s32le_planar -> s32p
pcm_s8 -> u8
pcm_s8_planar -> u8p
pcm_u16be -> s16
pcm_u16le -> s16
pcm_u24be -> s32
pcm_u24le -> s32
pcm_u32be -> s32
pcm_u32le -> s32
pcm_u8 -> u8
real_144 -> s16
roq_dpcm -> s16
s302m -> s32, s16
sonic -> s16
sonicls -> s16
tta -> u8, s16, s32
vorbis -> fltp
wavpack -> u8p, s16p, s32p, fltp
wmav1 -> fltp
wmav2 -> fltp
}
    LTemp := grpFormats.Items.Strings[grpFormats.ItemIndex];
    if SameText(LTemp, 'mp3') or (LAudioCodec = 'libmp3lame') then
      AppendOption('-sample_fmt s16p');
  end;

  // TODO: here you can add more options

  // NOTICE: the component will use output filename's FileExt to guess
  //         the output format when "-f" isn't specified.

  // change vcd/svcd/dvd file ext to .mpg
  if Pos(LFileExt, '.vcd.svcd.dvd') > 0 then
    LFileExt := '_' + LowerCase(grpNorm.Items.Strings[grpNorm.ItemIndex]) +
                '_' + Copy(LFileExt, 2, MaxInt) + '.mpg';

  // generate output filename automatically
  if edtOutPath.Text = SSameAsInput then
    LBasename := ChangeFileExt(FDecoder.FileName, '')
  else
    LBasename := edtOutPath.Text + ChangeFileExt(ExtractFileName(FDecoder.FileName), '');
  LFilename := LBasename + '_(new)' + LFileExt;
  if FileExists(LFilename) and not chkOverwrite.Checked then
  begin
    I := 1;
    while FileExists(LBasename + '_(new_' + IntToStr(I) + ')' + LFileExt) do
      Inc(I);
    LFilename := LBasename + '_(new_' + IntToStr(I) + ')' + LFileExt;
  end;

  if chkOverwrite.Checked then
    LOptions := LOptions + ' -y ';

  Result := LOptions + Format(' "%s"', [LFilename]);
end;

end.
