unit ConverterFrm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ComCtrls, ExtCtrls, Menus, FFBaseComponent, FFEncode,
  FFDecode, FFLoad, FFLog;

type
  TfrmConverter = class(TForm)
    btnAdd: TButton;
    btnRemove: TButton;
    btnClear: TButton;
    btnStart: TButton;
    btnStop: TButton;
    btnPause: TButton;
    btnResume: TButton;
    btnExit: TButton;
    btnWebSite: TButton;
    lvFiles: TListView;
    mmoLog: TMemo;
    grpOption: TGroupBox;
    Label1: TLabel;
    chkAsynchronous: TCheckBox;
    edtTasksCount: TEdit;
    chkPreview: TCheckBox;
    grpLogLevel: TRadioGroup;
    FFEncoder: TFFEncoder;
    FFDecoder: TFFDecoder;
    OpenDialog1: TOpenDialog;
    PopupMenu1: TPopupMenu;
    mnuAdd: TMenuItem;
    mnuRemove: TMenuItem;
    mnuClear: TMenuItem;
    N1: TMenuItem;
    mnuPlay: TMenuItem;
    mnuOpenFolder: TMenuItem;
    FFLogger: TFFLogger;
    procedure FormCreate(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormDestroy(Sender: TObject);
    procedure btnAddClick(Sender: TObject);
    procedure btnRemoveClick(Sender: TObject);
    procedure btnClearClick(Sender: TObject);
    procedure btnStartClick(Sender: TObject);
    procedure btnStopClick(Sender: TObject);
    procedure btnPauseClick(Sender: TObject);
    procedure btnResumeClick(Sender: TObject);
    procedure btnExitClick(Sender: TObject);
    procedure btnWebSiteClick(Sender: TObject);
    procedure chkPreviewClick(Sender: TObject);
    procedure PopupMenu1Popup(Sender: TObject);
    procedure mnuPlayClick(Sender: TObject);
    procedure mnuOpenFolderClick(Sender: TObject);
    procedure FFEncoderVideoInputHook(Sender: TObject; AHookInfo: PHookInfo);
    procedure FFEncoderVideoOutputHook(Sender: TObject; AHookInfo: PHookInfo);
    procedure FFEncoderAudioOutputHook(Sender: TObject; ATaskIndex: Integer;
      const APTS: Int64; AFrame: PAVFrame; ASize, ASampleRate,
      AChannels: Integer; ASampleFormat: TAVSampleFormat);
    procedure FFEncoderProgress(Sender: TObject; AProgressInfo: PProgressInfo);
    procedure FFEncoderTerminate(Sender: TObject; const ATerminateInfo: TTerminateInfo);
    procedure grpLogLevelClick(Sender: TObject);
    procedure FFLoggerLog(Sender: TObject; AThreadID: Integer;
      ALogLevel: TLogLevel; const ALogMsg: string);
  private
    FFolderList: TStringList;
    FOutFileList: TStringList;
    procedure DoAddTask(AOptions: string; AEventOptions: TEventOptions);
  public
  end;

var
  frmConverter: TfrmConverter;

implementation

{$R *.dfm}

uses
  libavutil_samplefmt, // for AV_SAMPLE_FMT_S16
  ShellAPI,
  OptionFrm,
  MyUtils;

var
  GfrmOption: TfrmOption = nil;

const
  CLibAVPath = 'LibAV';

  SAppTitle = 'Demo of FFEncoder %s';
  SCaption = 'Demo of FFEncoder %s - Delphi FFmpeg VCL Components';
  SWebSiteC = 'http://www.CCAVC.com';
  SWebSiteE = 'http://www.DelphiFFmpeg.com';
  DEMO_INFO =
    '* Asynchronous means creating new threads to perform converting tasks, ' +
      'it will perform tasks in a non-blocking way so that you can Stop/Pause/Resume ' +
      'tasks.' +
      #13#10 +
      #13#10;


  CDialogOptions = [ofHideReadOnly, ofFileMustExist, ofEnableSizing];
  CPictureFiles = '*.BMP;*.GIF;*.JPEG;*.JPG;*.PNG;';
  CAudioFiles = '*.AAC;*.AC3;*.APE;*.DTS;*.FLAC;*.M4A;*.MKA;*.MP2;*.MP3;' +
      '*.MPA;*.MPC;*.OFR;*.OGG;*.RA;*.TTA;*.WAV;*.WMA;';
  CVideoFiles = '*.3GP;*.ASF;*.AVI;*.AVM;*.AVS;*.DAT;*.FLV;*.MKV;*.MOV;' +
      '*.MP4;*.MPEG;*.MPG;*.NSV;*.OGM;*.RM;*.RMVB;*.TP;*.TS;*.VOB;*.WMV;';
  CDialogFilter =
      'Video/Audio/Picture Files|' + CVideoFiles + CAudioFiles + CPictureFiles +
      '|Video Files|' + CVideoFiles +
      '|Audio Files|' + CAudioFiles +
      '|Picture Files|' + CPictureFiles +
      '|All Files|*.*';

  SHookFrameNumber = 'FFEncoder - Frame Number: %d';
  SHookTimeStamp = 'FFEncoder - Time Stamp: %d';
  SHookTextImage = 'FFEncoder';
  SHookReverseVertical = 'Reverse Picture Vertically';
  SHookReverseHorizontal = 'Reverse Picture Horizontally';

var
  SWebSite: string = SWebSiteE;

procedure TfrmConverter.FormCreate(Sender: TObject);
begin
  // initialize
  Application.Title := Format(SAppTitle, [FFEncoder.Version]);
  Self.Caption := Format(SCaption, [FFEncoder.Version]);

  if SysUtils.SysLocale.PriLangID = LANG_CHINESE then
    SWebSite := SWebSiteC
  else
    SWebSite := SWebSiteE;
  btnWebSite.Caption := SWebSite;

  mmoLog.Text := DEMO_INFO;

  // ensure local decimal separator to be '.' and thousand separator to be ','
{$IF CompilerVersion >= 22.0}
  SysUtils.FormatSettings.DecimalSeparator := '.';
  SysUtils.FormatSettings.ThousandSeparator := ',';
{$ELSE}
  SysUtils.DecimalSeparator := '.';
  SysUtils.ThousandSeparator := ',';
{$IFEND}

  FFolderList := TStringList.Create;
  FOutFileList := TStringList.Create;

  // open dialog setting
  OpenDialog1.Options := CDialogOptions;
  OpenDialog1.Filter := CDialogFilter;

  // disable all fpu exceptions(floating point exceptions):
  //   invalid operation, denormalized, divide by zero, overflow, underflow, inexact/precision
  FFEncoder.DisableFPUExceptions;
end;

procedure TfrmConverter.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  with FFEncoder do
  begin
    // Clear the event handlers
    OnProgress := nil;
    OnTerminate := nil;
    OnVideoInputHook := nil;
    OnVideoOutputHook := nil;

    // Break converting
    Stop;
  end;
  FFLogger.OnLog := nil;
end;

procedure TfrmConverter.FormDestroy(Sender: TObject);
begin
  FreeAndNil(FFolderList);
  FreeAndNil(FOutFileList);
end;

procedure TfrmConverter.chkPreviewClick(Sender: TObject);
begin
  FFEncoder.PreviewVideo := chkPreview.Checked;
end;

procedure TfrmConverter.btnAddClick(Sender: TObject);
begin
  if not Assigned(GfrmOption) then
    // need to create option form
    GfrmOption := TfrmOption.Create(Self);

  // Load dynamic link libraries
  if not FFEncoder.AVLibLoaded then
  begin
    // TPathFileName = type WideString;
    // FFEncoder.LoadAVLib(const APath: TPathFileName): Boolean;
    // APath: Full path indicates location of FFmpeg DLLs.
    //        It can be empty, let Windows search DLLs in current dir or environment <PATH>
    //if not FFEncoder.LoadAVLib(ExtractFilePath(Application.ExeName) + CLibAVPath) then
    // the routine ExePath() is implemented in unit MyUtils which returns WideString type
    // of ExtractFilePath(Application.ExeName)
    if not FFEncoder.LoadAVLib(ExePath + CLibAVPath) then
    begin
      mmoLog.Lines.Add(FFEncoder.LastErrMsg);
      Exit;
    end;
  end;

  // open media file
  if not OpenDialog1.Execute then
    // cancel open file
    Exit;

  // now, try to load the input file selected to get AVFileInfo,
  // and then generate input options and output options for conversion

  if not FFDecoder.LoadFile(OpenDialog1.FileName) then
  begin // load av file failed
    mmoLog.Lines.Add('');
    mmoLog.Lines.Add('***File load error: ' + FFDecoder.LastErrMsg);
    mmoLog.Lines.Add('');
    Exit;
  end;

  try
    // init option form with FFDecoder
    GfrmOption.Decoder := FFDecoder;

    if GfrmOption.ShowModal = mrOk then
      // add task to conversion list
      DoAddTask(GfrmOption.Options, GfrmOption.EventOptions);
  finally
    // close the file handle in FFDecoder
    FFDecoder.CloseFile;
  end;
end;

procedure TfrmConverter.DoAddTask(AOptions: string; AEventOptions: TEventOptions);
var
  LIndex: Integer;
  LFilename: string;
begin
  // add a task
  LIndex := FFEncoder.AddTask(AOptions, @AEventOptions);
  if LIndex < 0 then
  begin
    mmoLog.Lines.Add('');
    mmoLog.Lines.Add('***Add task error: ' + FFEncoder.LastErrMsg);
    mmoLog.Lines.Add('');
    Exit;
  end;

  with lvFiles.Items.Add do
  begin
    // display some information of input file and output file
    LFilename := FFEncoder.InputFilenames[LIndex, 0];
    Caption := ExtractFileName(LFilename);
    // microseconds to seconds
    if FFDecoder.FileStreamInfo.Duration > 0 then
      SubItems.Add(IntToStr(FFDecoder.FileStreamInfo.Duration div 1000000))
    else
      SubItems.Add('N/A');
    SubItems.Add(IntToStr(FFDecoder.FileSize));
    LFilename := FFEncoder.OutputFilenames[LIndex, 0];
    SubItems.Add(ExtractFileName(LFilename));
    SubItems.Add('');
    SubItems.Add('');
    SubItems.Add('');
    SubItems.Add('');
  end;

  mmoLog.Lines.Add('');
  mmoLog.Lines.Add('***Task has been added to the list.');
  mmoLog.Lines.Add('');

  // set status of buttons
  btnRemove.Enabled := True;
  btnClear.Enabled := True;
  btnStart.Enabled := True;

  // add output file path to folder list
  FFolderList.Add(ExtractFilePath(LFilename));
  FOutFileList.AddObject(LFilename, nil);
end;

procedure TfrmConverter.btnRemoveClick(Sender: TObject);
begin
  if lvFiles.ItemIndex >= 0 then
  begin
    // remove input file
    FFEncoder.RemoveTask(lvFiles.ItemIndex);

    FFolderList.Delete(lvFiles.ItemIndex);
    FOutFileList.Delete(lvFiles.ItemIndex);

    lvFiles.Items.Delete(lvFiles.ItemIndex);

    // set status of buttons
    btnRemove.Enabled := lvFiles.Items.Count > 0;
    btnClear.Enabled := lvFiles.Items.Count > 0;
    btnStart.Enabled := lvFiles.Items.Count > 0;
  end;
end;

procedure TfrmConverter.btnClearClick(Sender: TObject);
begin
  // clear all input files
  FFEncoder.ClearTasks;

  FFolderList.Clear;
  FOutFileList.Clear;

  lvFiles.Items.Clear;

  // set status of buttons
  btnAdd.Enabled := True;
  btnRemove.Enabled := False;
  btnClear.Enabled := False;
  btnStart.Enabled := False;
end;

procedure TfrmConverter.btnStartClick(Sender: TObject);
begin
  // set status of buttons
  btnAdd.Enabled := False;
  btnRemove.Enabled := False;
  btnClear.Enabled := False;

  btnStart.Enabled := False;
  btnStop.Enabled := chkAsynchronous.Checked;
  btnPause.Enabled := chkAsynchronous.Checked;
  btnResume.Enabled := False;

  FFEncoder.PreviewVideo := chkPreview.Checked;

  // procedure Start(ATaskCount: Integer);
  // ATaskCount: >  0, perform tasks in asynchronous/non-blocking mode
  //             >  1, perform multiple tasks in the same time
  //             <= 0, perform tasks in synchronous/blocking mode
  if chkAsynchronous.Checked then
    // perform tasks in asynchronous/non-blocking mode
    FFEncoder.Start(StrToIntDef(edtTasksCount.Text, 1))
  else
    // perform tasks in synchronous/blocking mode
    FFEncoder.Start(0);
end;

procedure TfrmConverter.btnStopClick(Sender: TObject);
begin
  btnStop.Enabled := False;
  FFEncoder.Stop; // only works in asynchronous mode
end;

procedure TfrmConverter.btnPauseClick(Sender: TObject);
begin
  btnPause.Enabled := False;
  btnResume.Enabled := True;
  FFEncoder.Pause; // only works in asynchronous mode
end;

procedure TfrmConverter.btnResumeClick(Sender: TObject);
begin
  btnPause.Enabled := True;
  btnResume.Enabled := False;
  FFEncoder.Resume; // only works in asynchronous mode
end;

procedure TfrmConverter.btnWebSiteClick(Sender: TObject);
  function FromEXE: string;
  var
    S: string;
  begin
    S := ChangeFileExt(ExtractFileName(Application.ExeName), '');
    S := StringReplace(S, '[', '', [rfReplaceAll]);
    S := StringReplace(S, ']', '', [rfReplaceAll]);
    S := StringReplace(S, ' ', '_', [rfReplaceAll]);
    Result := '/?from=exe_' + S;
  end;
begin
  ShellExecute(Application.Handle, 'Open',   {Do not Localize}
    PChar(LowerCase(SWebSite + FromEXE)), '',
    PChar(ExtractFilePath(Application.ExeName)), 1);
end;

procedure TfrmConverter.btnExitClick(Sender: TObject);
begin
  Close;
end;

procedure TfrmConverter.PopupMenu1Popup(Sender: TObject);
var
  I: Integer;
begin
  mnuAdd.Enabled := btnAdd.Enabled;
  mnuRemove.Enabled := btnRemove.Enabled;
  mnuClear.Enabled := btnClear.Enabled;
  mnuOpenFolder.Enabled := lvFiles.Items.Count > 0;

  if (lvFiles.ItemIndex >= 0) and (lvFiles.ItemIndex < FFolderList.Count) then
    I := lvFiles.ItemIndex
  else
    I := FOutFileList.Count - 1;

  mnuPlay.Enabled := (I >= 0) and Assigned(FOutFileList.Objects[I]);
end;

procedure TfrmConverter.mnuPlayClick(Sender: TObject);
var
  I: Integer;
begin
  if (lvFiles.ItemIndex >= 0) and (lvFiles.ItemIndex < FFolderList.Count) then
    I := lvFiles.ItemIndex
  else
    I := FOutFileList.Count - 1;

  if (I >= 0) and Assigned(FOutFileList.Objects[I]) and FileExists(FOutFileList.Strings[I]) then
    ShellExecute(Application.Handle, 'Open',   {Do not Localize}
      PChar(FOutFileList.Strings[I]), '',
      PChar(ExtractFilePath(FOutFileList.Strings[I])), SW_SHOWNORMAL);
end;

procedure TfrmConverter.mnuOpenFolderClick(Sender: TObject);
var
  LPath: string;
begin
  if (lvFiles.ItemIndex >= 0) and (lvFiles.ItemIndex < FFolderList.Count) then
    LPath := FFolderList.Strings[lvFiles.ItemIndex]
  else if FFolderList.Count > 0 then
    LPath := FFolderList.Strings[FFolderList.Count - 1]
  else
    LPath := '';

  if (LPath <> '') and DirectoryExists(LPath) then
    ShellExecute(Application.Handle, 'Open', // {Do not Localize}
      PChar(LPath), nil,
      PChar(LPath), SW_SHOWDEFAULT);
end;

procedure TfrmConverter.FFEncoderVideoInputHook(Sender: TObject; AHookInfo: PHookInfo);
begin
  // OnVideoInputHook event handler, only triggered with OutputOptions.VideoInputHook = True
  // please refer to OnVideoOutputHook for detail
  if AHookInfo.FrameNumber = 1 then
  begin // first frame number
    with AHookInfo.Bitmap.Canvas.Font do
    begin // setup font example
      Color := clRed;
      Name := 'Tahoma';
      Size := 36;
      Style := [fsBold];
    end;
    // capture video frame example
    //AHookInfo.Bitmap.SaveToFile('C:\VideoHook_' + FloatToStr(Now) + '.bmp');
  end;
  if AHookInfo.FrameNumber < 550 then
    AHookInfo.Bitmap.Canvas.TextOut(0, 0, 'Video Input Hook')
  else
    AHookInfo.StopHook := True;
end;

procedure TfrmConverter.FFEncoderVideoOutputHook(Sender: TObject; AHookInfo: PHookInfo);
  function BytesPerPixel(APixelFormat: TPixelFormat): Integer;
  begin
    case APixelFormat of
      pf8bit: Result := 1;
      pf15bit, pf16bit: Result := 2;
      pf24bit: Result := 3;
      pf32bit: Result := 4;
    else
      raise Exception.Create('unsupported pixel format');
    end;
  end;
var
  H, W: Integer;
  I, J, K: Integer;
  P1, P2: PAnsiChar;
  B: AnsiChar;
  LBytes: Integer;
begin
  // OnVideoOutputHook event handler, only triggered with OutputOptions.VideoOutputHook = True
{
  PHookInfo = ^THookInfo;
  THookInfo = record
    TaskIndex: Integer;     // index of converting tasks
    Bitmap: TBitmap;        // bitmap filled with the original video picture, you can save it,
                            //  or change it by drawing text or image on bitmap.Canvas,
                            //  but you must NOT change the size and the PixelFormat of the bitmap!
    FrameNumber: Integer;   // frame index number, first is 1 not 0
    PTS: Int64;             // presentation time stamp of current picture, in microseconds
    Update: Boolean;        // whether update the bitmap back to original video picture, default True
    StopHook: Boolean;      // whether stop video hook, if true then VideoHook event will not
                            //  be triggered with this file, default False
  end;
}
  if AHookInfo.FrameNumber = 1 then
  begin // first frame number
    with AHookInfo.Bitmap.Canvas.Font do
    begin // setup font example
      Color := clWhite;
      Name := 'Tahoma';
      Size := 12;
      Style := [fsBold, fsUnderline];
    end;
  end;

  if AHookInfo.FrameNumber < 100 then
    // text out with frame number example
    AHookInfo.Bitmap.Canvas.TextOut(10, 10, Format(SHookFrameNumber, [AHookInfo.FrameNumber]))
  else if AHookInfo.FrameNumber < 150 then
    // do not change the original video picture example
    AHookInfo.Update := False
  else if AHookInfo.FrameNumber < 250 then
    // text out with Presentation Time Stamp example
    AHookInfo.Bitmap.Canvas.TextOut(10, 10, Format(SHookTimeStamp, [AHookInfo.PTS]))
  else if AHookInfo.FrameNumber < 350 then
  begin
    // draw text and graphic example
    if AHookInfo.FrameNumber = 250 then
      with AHookInfo.Bitmap.Canvas.Font do
      begin // change font example
        Color := clRed;
        Size := 16;
      end;
    AHookInfo.Bitmap.Canvas.TextOut(10, 10, SHookTextImage);
    AHookInfo.Bitmap.Canvas.Draw(AHookInfo.Bitmap.Width - Application.Icon.Width - 10,
                        AHookInfo.Bitmap.Height - Application.Icon.Height - 10,
                        Application.Icon);
  end
  else if AHookInfo.FrameNumber < 450 then
  begin
    LBytes := BytesPerPixel(AHookInfo.Bitmap.PixelFormat);
    // reverse picture vertically example
    H := AHookInfo.Bitmap.Height;
    W := AHookInfo.Bitmap.Width;
    for I := 0 to H div 2 - 1 do
    begin
      P1 := PAnsiChar(AHookInfo.Bitmap.ScanLine[I]);
      P2 := PAnsiChar(AHookInfo.Bitmap.ScanLine[H - 1 - I]);
      for J := 0 to LBytes * W - 1 do
      begin
        B := P1^;
        P1^ := P2^;
        P2^ := B;
        Inc(P1);
        Inc(P2);
      end;
    end;
    if AHookInfo.FrameNumber = 350 then
      with AHookInfo.Bitmap.Canvas.Font do
      begin // change font example
        Color := clWhite;
        Style := [fsBold];
      end;
    AHookInfo.Bitmap.Canvas.TextOut(10, 10, SHookReverseVertical);
  end
  else if AHookInfo.FrameNumber < 550 then
  begin
    LBytes := BytesPerPixel(AHookInfo.Bitmap.PixelFormat);
    // reverse picture horizontally example
    H := AHookInfo.Bitmap.Height;
    W := AHookInfo.Bitmap.Width;
    for I := 0 to H - 1 do
    begin
      P1 := PAnsiChar(AHookInfo.Bitmap.ScanLine[I]);
      P2 := P1 + LBytes * (W - 1);
      for J := 0 to W div 2 - 1 do
      begin
        for K := 0 to LBytes - 1 do
        begin
          B := (P1 + K)^;
          (P1 + K)^ := (P2 + K)^;
          (P2 + K)^ := B;
        end;
        Inc(P1, LBytes);
        Dec(P2, LBytes);
      end;
    end;
    if AHookInfo.FrameNumber = 350 then
      with AHookInfo.Bitmap.Canvas.Font do
      begin // change font example
        Style := [];
      end;
    AHookInfo.Bitmap.Canvas.TextOut(10, 10, SHookReverseHorizontal);
  end
  else
    // stop video hook exmaple, the converting will speedup
    AHookInfo.StopHook := True;
end;

procedure TfrmConverter.FFEncoderAudioOutputHook(Sender: TObject;
  ATaskIndex: Integer; const APTS: Int64; AFrame: PAVFrame; ASize, ASampleRate,
  AChannels: Integer; ASampleFormat: TAVSampleFormat);
var
  P: PSmallInt;
  I, C: Integer;
begin
  // OnAudioOutputHook event handler, only triggered with OutputOptions.AudioOutputHook = True
  // example for disabling left channel or right channel, toggle every 5 seconds
  //Assert(AFrame.format = Ord(ASampleFormat));
  if AChannels = 1 then
  begin
    (Sender as TFFEncoder).FFmpegs[ATaskIndex].OnOutputAudioHook := nil;
    mmoLog.Lines.Add('***Ignore mono for current audio hooking.');
  end
  else if ASampleFormat = AV_SAMPLE_FMT_S16 then
  begin
    // signed 16 bits
    P := PSmallInt(AFrame.data[0]);
    for I := 0 to ASize div SizeOf(SmallInt) div AChannels - 1 do
    begin
      if APTS div 5000000 mod 2 = 0 then
      begin
        for C := 0 to AChannels - 1 do
        begin
          if (C mod 2) = 0 then
            P^ := 0;    // disable left channel
            //P^ := PSmallInt(PAnsiChar(P) + SizeOf(SmallInt))^;  // copy right channel to left channel
          Inc(P);
        end;
      end
      else
      begin
        for C := 0 to AChannels - 1 do
        begin
          if (C mod 2) = 1 then
            P^ := 0;    // disable right channel
            //P^ := PSmallInt(PAnsiChar(P) - SizeOf(SmallInt))^;  // copy left channel to right channel
          Inc(P);
        end;
      end;
    end;
    if APTS div 5000000 mod 2 = 0 then
    begin
      for C := 0 to ASize div SizeOf(SmallInt) - ASize div SizeOf(SmallInt) div AChannels * AChannels - 1 do
      begin
        if (C mod 2) = 0 then
          P^ := 0; // disable left channel
          //P^ := PSmallInt(PAnsiChar(P) + SizeOf(SmallInt))^;  // copy right channel to left channel
        Inc(P);
      end;
    end
    else
    begin
      for C := 0 to ASize div SizeOf(SmallInt) - ASize div SizeOf(SmallInt) div AChannels * AChannels - 1 do
      begin
        if (C mod 2) = 1 then
          P^ := 0; // disable right channel
          //P^ := PSmallInt(PAnsiChar(P) - SizeOf(SmallInt))^;  // copy left channel to right channel
        Inc(P);
      end;
    end;
  end
  else if ASampleFormat = AV_SAMPLE_FMT_S16P then
  begin
    // signed 16 bits, planar
    if APTS div 5000000 mod 2 = 0 then
    begin
      for C := 0 to AChannels - 1 do
      begin
        if (C mod 2) = 0 then
          FillChar(AFrame.data[C]^, AFrame.linesize[0], 0); // disable left channel
          //if C < AChannels - 1 then
          //  Move(AFrame.data[C + 1]^, AFrame.data[C]^, AFrame.linesize[0])  // copy right channel to left channel
          //else
          //  FillChar(AFrame.data[C]^, AFrame.linesize[0], 0); // disable left channel
      end;
    end
    else
    begin
      for C := 0 to AChannels - 1 do
      begin
        if (C mod 2) = 1 then
          FillChar(AFrame.data[C]^, AFrame.linesize[0], 0); // disable right channel
          //Move(AFrame.data[C - 1]^, AFrame.data[C]^, AFrame.linesize[0])  // copy right channel to left channel
      end;
    end;
  end
  else
  begin
    (Sender as TFFEncoder).FFmpegs[ATaskIndex].OnOutputAudioHook := nil;
    mmoLog.Lines.Add('***Unsupported sample format for current audio hooking.');
  end;
end;

procedure TfrmConverter.FFEncoderProgress(Sender: TObject; AProgressInfo: PProgressInfo);
begin
  // OnProgress event handler
{
  PProgressInfo = ^TProgressInfo;
  TProgressInfo = record
    TaskIndex: Integer;     // index of converting tasks
    FileIndex: Integer;     // index of input files in the current task
    FrameNumber: Integer;   // current frame number
    FPS: Integer;           // video converting speed, frames per second, not valid when only audio output
    Quality: Single;        // quality
    BitRate: Single;        // bitrate
    CurrentSize: Int64;     // current output file size in bytes
    CurrentDuration: Int64; // current duration time in microsecond
    TotalDuration: Int64;   // total output duration time in microsecond
  end;
}
  with lvFiles.Items.Item[AProgressInfo.TaskIndex].SubItems, AProgressInfo^ do
  begin
    if TotalDuration > 0 then
      Strings[3] := IntToStr(CurrentDuration * 100 div TotalDuration) + '%'
    else
      Strings[3] := 'N/A';

    if FPS > 0 then
      Strings[4] := IntToStr(FPS);

    Strings[5] := IntToStr(CurrentDuration div 1000000); // display as seconds
    Strings[6] := IntToStr(CurrentSize);
  end;

  if not chkAsynchronous.Checked then
    // make sure update converting status when converting in synchronous/blocking mode
    lvFiles.Repaint;
end;

procedure TfrmConverter.FFEncoderTerminate(Sender: TObject; const ATerminateInfo: TTerminateInfo);
begin
  // OnTerminate event handler
{
  TTerminateInfo = record
    TaskIndex: Integer;     // index of converting tasks, (-1) means all tasks are terminated
    Finished: Boolean;      // True means converted success, False means converting broken
    Exception: Boolean;     // True means Exception occured, False please ignore
    ExceptionMsg: string;   // Exception message
  end;
}
  if ATerminateInfo.TaskIndex < 0 then
  begin
    // set status of buttons
    btnAdd.Enabled := False;
    btnRemove.Enabled := False;
    btnClear.Enabled := True;
    btnStart.Enabled := False;
    btnStop.Enabled := False;
    btnPause.Enabled := False;
    btnResume.Enabled := False;
  end
  else if ATerminateInfo.Finished then
  begin // TaskIndex task converted success
    lvFiles.Items.Item[ATerminateInfo.TaskIndex].SubItems.Strings[3] := '100%';
    if not chkAsynchronous.Checked then
    begin // make sure update converting status when converting synchronous/blocking mode
      lvFiles.Repaint;
    end;
  end;

  if (ATerminateInfo.TaskIndex >= 0) and not ATerminateInfo.Exception then
    // indicate the output file is ready to play
    FOutFileList.Objects[ATerminateInfo.TaskIndex] := TObject(1);

  if ATerminateInfo.Exception then // Exception occured, show exception message
    Application.MessageBox(PChar(ATerminateInfo.ExceptionMsg), PChar(Application.Title), MB_OK + MB_ICONWARNING);
end;

procedure TfrmConverter.grpLogLevelClick(Sender: TObject);
begin
  // TLogLevel = (llQuiet, llPanic, llFatal, llError, llWarning, llInfo, llVerbose, llDebug, llTrace);
  FFLogger.LogLevel := TLogLevel(grpLogLevel.ItemIndex);
end;

procedure TfrmConverter.FFLoggerLog(Sender: TObject; AThreadID: Integer;
  ALogLevel: TLogLevel; const ALogMsg: string);
begin
  mmoLog.Lines.Add(ALogMsg);
end;

end.
