unit AudioVideoMergerFrm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, FFBaseComponent, FFEncode, FFDecode, FFLoad, FFLog;

type
  TfrmAudioVideoMerger = class(TForm)
    Label1: TLabel;
    lblStatus: TLabel;
    btnOpen: TButton;
    btnStart: TButton;
    btnStop: TButton;
    btnPause: TButton;
    btnResume: TButton;
    btnWebSite: TButton;
    cboLogLevel: TComboBox;
    chkPreview: TCheckBox;
    btnVideo: TButton;
    txtVideo: TStaticText;
    btnAudio: TButton;
    txtAudio: TStaticText;
    txtOutput: TStaticText;
    mmoLog: TMemo;
    FFEncoder: TFFEncoder;
    FFDecoder: TFFDecoder;
    FFLogger: TFFLogger;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure btnVideoClick(Sender: TObject);
    procedure btnAudioClick(Sender: TObject);
    procedure btnOpenClick(Sender: TObject);
    procedure btnStartClick(Sender: TObject);
    procedure btnStopClick(Sender: TObject);
    procedure btnPauseClick(Sender: TObject);
    procedure btnResumeClick(Sender: TObject);
    procedure btnWebSiteClick(Sender: TObject);
    procedure cboLogLevelChange(Sender: TObject);
    procedure chkPreviewClick(Sender: TObject);
    procedure FFEncoderProgress(Sender: TObject; AProgressInfo: PProgressInfo);
    procedure FFEncoderTerminate(Sender: TObject; const ATerminateInfo: TTerminateInfo);
    procedure FFLoggerLog(Sender: TObject; AThreadID: Integer;
      ALogLevel: TLogLevel; const ALogMsg: string);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  frmAudioVideoMerger: TfrmAudioVideoMerger;

implementation

{$R *.dfm}

uses
  ShellAPI,
  MyUtils;

const
  CLibAVPath = 'LibAV';

  SAppTitle = 'Demo of FFEncoder %s';
  SCaption = 'Demo of FFEncoder %s (AudioVideoMerger) - Delphi FFmpeg VCL Components';
  SWebSiteC = 'http://www.CCAVC.com';
  SWebSiteE = 'http://www.DelphiFFmpeg.com';


  CDialogOptions = [ofHideReadOnly, ofFileMustExist, ofEnableSizing];
  CAudioFiles = '*.AAC;*.AC3;*.APE;*.DTS;*.FLAC;*.M4A;*.MKA;*.MP2;*.MP3;' +
      '*.MPA;*.MPC;*.OFR;*.OGG;*.RA;*.TTA;*.WAV;*.WMA;';
  CVideoFiles = '*.3GP;*.ASF;*.AVI;*.AVM;*.AVS;*.DAT;*.FLV;*.MKV;*.MOV;' +
      '*.MP4;*.MPEG;*.MPG;*.NSV;*.OGM;*.RM;*.RMVB;*.TP;*.TS;*.VOB;*.WMV;';

var
  SWebSite: string = SWebSiteE;

procedure TfrmAudioVideoMerger.FormCreate(Sender: TObject);
begin
  Application.Title := Format(SAppTitle, [FFEncoder.Version]);
  Self.Caption := Format(SCaption, [FFEncoder.Version]);

  if SysUtils.SysLocale.PriLangID = LANG_CHINESE then
    SWebSite := SWebSiteC
  else
    SWebSite := SWebSiteE;

  mmoLog.Text := SWebSite + #13#10#13#10;
  btnWebsite.Hint := SWebSite;
  btnWebsite.ShowHint := True;

  // open dialog setting
  OpenDialog1.Options := CDialogOptions;

  // save dialog setting
  SaveDialog1.Options := [ofOverwritePrompt, ofHideReadOnly, ofPathMustExist, ofEnableSizing];
  SaveDialog1.Filter := 'MP4 format(*.mp4)|*.mp4';
  SaveDialog1.DefaultExt := 'mp4';

  // disable all fpu exceptions(floating point exceptions):
  //   invalid operation, denormalized, divide by zero, overflow, underflow, inexact/precision
  FFEncoder.DisableFPUExceptions;
end;

procedure TfrmAudioVideoMerger.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  with FFEncoder do
  begin
    // Clear the event handlers
    OnProgress := nil;
    OnTerminate := nil;

    // Break converting
    Stop;
  end;
  FFLogger.OnLog := nil;
end;

procedure TfrmAudioVideoMerger.btnVideoClick(Sender: TObject);
begin
  OpenDialog1.Filter := 'Video Files|' + CVideoFiles;
  OpenDialog1.FileName := '';
  if OpenDialog1.Execute then
    txtVideo.Caption := OpenDialog1.FileName;
  btnOpen.Enabled := FileExists(txtVideo.Caption) and FileExists(txtAudio.Caption);
end;

procedure TfrmAudioVideoMerger.btnAudioClick(Sender: TObject);
begin
  OpenDialog1.Filter := 'Audio/Video Files|' + CAudioFiles + CVideoFiles;
  OpenDialog1.FileName := '';
  if OpenDialog1.Execute then
    txtAudio.Caption := OpenDialog1.FileName;
  btnOpen.Enabled := FileExists(txtVideo.Caption) and FileExists(txtAudio.Caption);
end;

procedure TfrmAudioVideoMerger.btnOpenClick(Sender: TObject);
begin
  // Load dynamic link libraries
  if not FFEncoder.AVLibLoaded then
  begin
    // TPathFileName = type WideString;
    // FFEncoder.LoadAVLib(const APath: TPathFileName): Boolean;
    // APath: Full path indicates location of FFmpeg DLLs.
    //        It can be empty, let Windows search DLLs in current dir or environment <PATH>
    //if not FFEncoder.LoadAVLib(ExtractFilePath(Application.ExeName) + CLibAVPath) then
    // the routine ExePath() is implemented in unit MyUtils which returns WideString type
    // of ExtractFilePath(Application.ExeName)
    if not FFEncoder.LoadAVLib(ExePath + CLibAVPath) then
    begin
      mmoLog.Lines.Add(FFEncoder.LastErrMsg);
      Exit;
    end;
  end;

  // output filename
  SaveDialog1.FileName := Format('%s+%s.mp4',
                                [ChangeFileExt(ExtractFileName(txtVideo.Caption), ''),
                                ChangeFileExt(ExtractFileName(txtAudio.Caption), '')]);
  if SaveDialog1.Execute then
  begin
    txtOutput.Caption := SaveDialog1.FileName;
    btnStart.Enabled := True;
    btnStart.SetFocus;
  end;
end;

procedure TfrmAudioVideoMerger.btnStartClick(Sender: TObject);
var
  LIndex: Integer;
  LMapping: string;
  LOptions: string;
  LVideoFileName: string;
  LVideoStreamIndex: Integer;
  LAudioFileName: string;
  LAudioStreamIndex: Integer;
  LAudioStreamIndex2: Integer;
begin
  // ensure reset FFEncoder
  FFEncoder.ClearTasks;
  mmoLog.Lines.Add('');

  // check video input file
  LVideoFileName := txtVideo.Caption;
  if not FFDecoder.LoadFile(LVideoFileName) then
  begin // can't load
    mmoLog.Lines.Add('');
    mmoLog.Lines.Add('***Video file open error: ' + FFDecoder.LastErrMsg);
    mmoLog.Lines.Add('');
    Exit;
  end;
  mmoLog.Lines.Add(FFDecoder.FileInfoText);
  LVideoStreamIndex := FFDecoder.FirstVideoStreamIndex;
  LAudioStreamIndex2 := FFDecoder.FirstAudioStreamIndex;
  FFDecoder.CloseFile;
  if LVideoStreamIndex < 0 then
  begin // no video stream
    mmoLog.Lines.Add('');
    mmoLog.Lines.Add('***No video stream');
    mmoLog.Lines.Add('');
    Exit;
  end;

  // check audio input file
  LAudioFileName := txtAudio.Caption;
  if not FFDecoder.LoadFile(LAudioFileName) then
  begin // can't load
    mmoLog.Lines.Add('');
    mmoLog.Lines.Add('***Audio file open error: ' + FFDecoder.LastErrMsg);
    mmoLog.Lines.Add('');
    Exit;
  end;
  mmoLog.Lines.Add(FFDecoder.FileInfoText);
  LAudioStreamIndex := FFDecoder.FirstAudioStreamIndex;
  FFDecoder.CloseFile;
  if LAudioStreamIndex < 0 then
  begin // no audio stream
    mmoLog.Lines.Add('');
    mmoLog.Lines.Add('***No audio stream');
    mmoLog.Lines.Add('');
    Exit;
  end;

  // important options for merging video and audio
  // -map file:stream[,syncfile:syncstream]  set input stream mapping
  LMapping := Format('-map 0:%d -map 1:%d', [LVideoStreamIndex, LAudioStreamIndex]);

  // example for second audio
  if LAudioStreamIndex2 >= 0 then
    LMapping := LMapping + Format(' -map 0:%d %s ', [LAudioStreamIndex2,
                                  '-c:a:1 mp2 -b:a:1 64k -ar:2 22050']);

  LOptions := Format('-i "%s" -i "%s" %s %s "%s"', [LVideoFileName, LAudioFileName,
                    '-c:a aac -b:a 128k -ar 44100 -c:v mpeg4 -s 320x240 -b:v 320k',
                    LMapping, txtOutput.Caption]);

  // add a task
  LIndex := FFEncoder.AddTask(LOptions);
  if LIndex < 0 then
  begin
    mmoLog.Lines.Add('');
    mmoLog.Lines.Add('***Add task error: ' + FFEncoder.LastErrMsg);
    mmoLog.Lines.Add('');
    Exit;
  end;

  mmoLog.Lines.Add('');
  mmoLog.Lines.Add('***Ready to go!');
  mmoLog.Lines.Add('');

  // set status of buttons
  btnOpen.Enabled := False;
  btnStart.Enabled := False;
  btnStop.Enabled := True;
  btnPause.Enabled := True;
  btnResume.Enabled := False;
  btnVideo.Enabled := False;
  btnAudio.Enabled := False;
  btnStop.SetFocus;

  FFEncoder.PreviewVideo := chkPreview.Checked;

  // procedure Start(ATaskCount: Integer);
  // ATaskCount: >  0, perform tasks in asynchronous/non-blocking mode
  //             >  1, perform multiple tasks in the same time
  //             <= 0, perform tasks in synchronous/blocking mode
  FFEncoder.Start(1);
end;

procedure TfrmAudioVideoMerger.btnStopClick(Sender: TObject);
begin
  btnStop.Enabled := False;
  FFEncoder.Stop; // only works in asynchronous mode
end;

procedure TfrmAudioVideoMerger.btnPauseClick(Sender: TObject);
begin
  btnPause.Enabled := False;
  btnResume.Enabled := True;
  FFEncoder.Pause; // only works in asynchronous mode
end;

procedure TfrmAudioVideoMerger.btnResumeClick(Sender: TObject);
begin
  btnPause.Enabled := True;
  btnResume.Enabled := False;
  FFEncoder.Resume; // only works in asynchronous mode
end;

procedure TfrmAudioVideoMerger.btnWebSiteClick(Sender: TObject);
  function FromEXE: string;
  var
    S: string;
  begin
    S := ChangeFileExt(ExtractFileName(Application.ExeName), '');
    S := StringReplace(S, '[', '', [rfReplaceAll]);
    S := StringReplace(S, ']', '', [rfReplaceAll]);
    S := StringReplace(S, ' ', '_', [rfReplaceAll]);
    Result := '/?from=exe_' + S;
  end;
begin
  ShellExecute(Application.Handle, 'Open',   {Do not Localize}
    PChar(LowerCase(SWebSite + FromEXE)), '',
    PChar(ExtractFilePath(Application.ExeName)), 1);
end;

procedure TfrmAudioVideoMerger.cboLogLevelChange(Sender: TObject);
begin
  FFLogger.LogLevel := TLogLevel(cboLogLevel.ItemIndex);
end;

procedure TfrmAudioVideoMerger.chkPreviewClick(Sender: TObject);
begin
  FFEncoder.PreviewVideo := chkPreview.Checked;
end;

procedure TfrmAudioVideoMerger.FFEncoderProgress(Sender: TObject;
  AProgressInfo: PProgressInfo);
begin
  // OnProgress event handler
{
  PProgressInfo = ^TProgressInfo;
  TProgressInfo = record
    TaskIndex: Integer;     // index of converting tasks
    FileIndex: Integer;     // index of input files in the current task
    FrameNumber: Integer;   // current frame number
    FPS: Integer;           // video converting speed, frames per second, not valid when only audio output
    Quality: Single;        // quality
    BitRate: Single;        // bitrate
    CurrentSize: Int64;     // current output file size in bytes
    CurrentDuration: Int64; // current duration time in microsecond
    TotalDuration: Int64;   // total output duration time in microsecond
  end;
}
  with AProgressInfo^ do
    lblStatus.Caption := Format('Frame number: %d; FPS: %d; Size: %d; Time: %d',
        [FrameNumber, FPS, CurrentSize, CurrentDuration]);
end;

procedure TfrmAudioVideoMerger.FFEncoderTerminate(Sender: TObject;
  const ATerminateInfo: TTerminateInfo);
begin
  // OnTerminate event handler
{
  TTerminateInfo = record
    TaskIndex: Integer;     // index of converting tasks, (-1) means all tasks are terminated
    Finished: Boolean;      // True means converted success, False means converting broken
    Exception: Boolean;     // True means Exception occured, False please ignore
    ExceptionMsg: string;   // Exception message
  end;
}
  if ATerminateInfo.TaskIndex < 0 then
  begin
    // set status of buttons
    btnStart.Enabled := False;
    btnStop.Enabled := False;
    btnPause.Enabled := False;
    btnResume.Enabled := False;
    btnVideo.Enabled := True;
    btnAudio.Enabled := True;
  end;
  if ATerminateInfo.Exception then // Exception occured, show exception message
    Application.MessageBox(PChar(ATerminateInfo.ExceptionMsg), PChar(Application.Title), MB_OK + MB_ICONWARNING);
end;

procedure TfrmAudioVideoMerger.FFLoggerLog(Sender: TObject; AThreadID: Integer;
  ALogLevel: TLogLevel; const ALogMsg: string);
begin
  mmoLog.Lines.Add(ALogMsg);
end;

end.
